package jcircus.visitor;

import java.io.StringWriter;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import jcircus.environment.ProcChanUseEnv;
import jcircus.environment.ChanSyncEnv;
import jcircus.environment.ChanUseEnv;
import jcircus.environment.Environment;
import jcircus.environment.ChanInfoEnv;
import jcircus.environment.ProcChanEnv;
import jcircus.environment.NameTypeEnv;
import jcircus.exceptions.ChannelNotDefinedExceptionForProcess;
import jcircus.exceptions.InvalidParameterException;
import jcircus.exceptions.InvalidSubTypeException;
import jcircus.exceptions.NotYetImplementedException;
import jcircus.exceptions.UnrecoveredErrorException;
import jcircus.exceptions.VelocityException;
import jcircus.exceptions.VisitingMethodNotDefinedException;
import jcircus.exceptions.JCircusException;
import jcircus.util.CircusType;
import jcircus.util.Constants;
import jcircus.util.ChanUse;
import jcircus.util.Error;
import jcircus.util.ErrorMessage;
import jcircus.util.NameType;
import jcircus.util.MathToolkitConstants;
import jcircus.util.SignatureExpression;
import jcircus.util.ChanSync;
import jcircus.util.Util;
import jcircus.util.annotations.IdCircusProcessAnn;
import net.sourceforge.czt.base.ast.ListTerm;
import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.ast.TermA;
import net.sourceforge.czt.base.impl.ListTermImpl;
import net.sourceforge.czt.circus.ast.ActionD;
import net.sourceforge.czt.circus.ast.ActionIte;
import net.sourceforge.czt.circus.ast.ActionPara;
import net.sourceforge.czt.circus.ast.AssignmentCommand;
import net.sourceforge.czt.circus.ast.AssignmentPair;
import net.sourceforge.czt.circus.ast.BasicNameSet;
import net.sourceforge.czt.circus.ast.BasicProcess;
import net.sourceforge.czt.circus.ast.CallAction;
import net.sourceforge.czt.circus.ast.CallProcess;
import net.sourceforge.czt.circus.ast.CallType;
import net.sourceforge.czt.circus.ast.ChaosAction;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.CircusProcess;
import net.sourceforge.czt.circus.ast.Communication;
import net.sourceforge.czt.circus.ast.DotField;
import net.sourceforge.czt.circus.ast.ExtChoiceAction;
import net.sourceforge.czt.circus.ast.ExtChoiceProcess;
import net.sourceforge.czt.circus.ast.Field;
import net.sourceforge.czt.circus.ast.GuardedAction;
import net.sourceforge.czt.circus.ast.HideProcess;
import net.sourceforge.czt.circus.ast.IfGuardedCommand;
import net.sourceforge.czt.circus.ast.InputField;
import net.sourceforge.czt.circus.ast.IntChoiceAction;
import net.sourceforge.czt.circus.ast.IntChoiceActionIte;
import net.sourceforge.czt.circus.ast.IntChoiceProcess;
import net.sourceforge.czt.circus.ast.IntChoiceProcessIte;
import net.sourceforge.czt.circus.ast.MuAction;
import net.sourceforge.czt.circus.ast.NameSet;
import net.sourceforge.czt.circus.ast.OutputField;
import net.sourceforge.czt.circus.ast.ParAction;
import net.sourceforge.czt.circus.ast.ParProcess;
import net.sourceforge.czt.circus.ast.ParProcessIte;
import net.sourceforge.czt.circus.ast.ParamAction;
import net.sourceforge.czt.circus.ast.ParamProcess;
import net.sourceforge.czt.circus.ast.PrefixingAction;
import net.sourceforge.czt.circus.ast.Process2;
import net.sourceforge.czt.circus.ast.ProcessIte;
import net.sourceforge.czt.circus.ast.ProcessPara;
import net.sourceforge.czt.circus.ast.RenameProcess;
import net.sourceforge.czt.circus.ast.SchExprAction;
import net.sourceforge.czt.circus.ast.SeqAction;
import net.sourceforge.czt.circus.ast.SeqActionIte;
import net.sourceforge.czt.circus.ast.SeqProcess;
import net.sourceforge.czt.circus.ast.SeqProcessIte;
import net.sourceforge.czt.circus.ast.SkipAction;
import net.sourceforge.czt.circus.ast.StopAction;
import net.sourceforge.czt.circus.ast.VarDeclCommand;
import net.sourceforge.czt.z.ast.AndPred;
import net.sourceforge.czt.z.ast.ApplExpr;
import net.sourceforge.czt.z.ast.AxPara;
import net.sourceforge.czt.z.ast.Box;
import net.sourceforge.czt.z.ast.ConstDecl;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.DeclName;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.ExprPred;
import net.sourceforge.czt.z.ast.FalsePred;
import net.sourceforge.czt.z.ast.IffPred;
import net.sourceforge.czt.z.ast.ImpliesPred;
import net.sourceforge.czt.z.ast.MemPred;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.ast.NameTypePair;
import net.sourceforge.czt.z.ast.NegPred;
import net.sourceforge.czt.z.ast.NumExpr;
import net.sourceforge.czt.z.ast.Op;
import net.sourceforge.czt.z.ast.OrPred;
import net.sourceforge.czt.z.ast.Para;
import net.sourceforge.czt.z.ast.PowerExpr;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.ProdExpr;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.RefName;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.SetExpr;
import net.sourceforge.czt.z.ast.Signature;
import net.sourceforge.czt.z.ast.TruePred;
import net.sourceforge.czt.z.ast.TupleExpr;
import net.sourceforge.czt.z.ast.VarDecl;
import net.sourceforge.czt.z.util.Factory;
import net.sourceforge.czt.z.util.ZString;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import circparser.formw.action.Action;
import net.sourceforge.czt.z.ast.SchText;

/**
 * TranslatorVisitor.java
 * 
 * Generates Java code.
 *
 * @author Angela Freitas
 *
 */
public class TranslatorVisitor implements TranslatorVisitorInterface {
    
    private static final String LEFT = "left";
    private static final String RIGHT = "right";
    
    private static final int MIN = 0;
    private static final int MAX = 1;
    
    private static final int DECL_SCHEMA = 1;
    private static final int DECL_AXDEF = 2;
    
    private boolean externalChoice_ = false;
        
    private DeclName currentProcess_ = null;

    private DeclName currentAction_ = null;
    
    private Environment environment_;
    
    private String currentMuActionName_;
           
    /** Reinitialized in each action definition */
    private int index_;
    
    /** Reinitialized in each recursive action */
    private int newIndex_;
    
    /** Indicates if a declaration will be private */
    private boolean isPrivate_;
    
    private Factory factory_;
    
    private List<Error> errors_;
    
    public List<Error> getErrors() {
        return this.errors_;
    }
    
    /**
     * Methods to report errors.
     */
    
    private void reportError(Error error) {
        errors_.add(error);
    }
    
    private void reportFatalError(Error error) throws RuntimeException {
        reportError(error);
        throw new UnrecoveredErrorException("Error during the environment loading.");
    }
    
    /**
     * Constructor.
     *
     */
    public TranslatorVisitor(Environment environment) {
        this.index_ = -1;
        this.environment_ = environment;
        this.currentProcess_ = null;
        this.factory_ = new Factory();
        this.errors_ = new ArrayList();
    }
    
    /**
     * Just in case this visitor is called for a node that has not been considered yet.
     *
     * @param term
     * @return
     */
    public Object visitTerm(Term term) {
        
        throw new VisitingMethodNotDefinedException(term.getClass(), this.getClass());
    }
    
    
    /** **************************************************************************************************************
     * Para Tree.
     * ***************************************************************************************************************
     */

    /**
     * This may be a schema definition, axiomatic definition, or horizontal definition.
     */
    public Object visitAxPara(AxPara axPara) { 
        
        String code = null;
        
        DeclName varName;
        
        List gens = axPara.getDeclName();
        SchText schText = axPara.getSchText();
        Box box = axPara.getBox();
        
        if (box.equals(Box.AxBox)) {
            
            // Axiomatic definition
            List decl = schText.getDecl();
            Pred pred = schText.getPred();
            
            if (decl.size() == 1) {

                VarDecl varDecl = (VarDecl) decl.get(0);
                
                List names = varDecl.getDeclName();
                Expr expr = varDecl.getExpr();
                
                if (names.size() > 1)
                    reportError(new Error(ErrorMessage.AXDEF_ONE_DECL, new Object[]{}));
                
                // Name of the declared variable
                varName = (DeclName) names.get(0);
                
                CircusType circusType = Util.getCircusTypeAnn(varName);
                NameType nameType = Util.getNameTypeAnn(varName);
                
                // Find out if the axdef was declared in global scope or within a basic process
                boolean globalScope;
                if (nameType.equals(NameType.GlobalConstant)) {
                    globalScope = true;
                } else {
                    globalScope = false;
                }
                
                code = this.axDefCode(varName.toString(), circusType, pred, globalScope);
                
            } else {/*!(decl.size() == 1)*/
                reportError(new Error(ErrorMessage.AXDEF_ONE_DECL, new Object[]{}));
            }
            
        } else if (box.equals(Box.OmitBox)) {
            
            // Horizontal definition
            List decl = schText.getDecl();
            Pred pred = schText.getPred();
            
            ConstDecl constDecl = (ConstDecl) decl.get(0);

            DeclName declName = constDecl.getDeclName();
            Expr expr = constDecl.getExpr();
            
            CircusType circusType = Util.getCircusTypeAnn(declName);
            NameType nameType = Util.getNameTypeAnn(declName);
            
            // Find out if the horizDef was declared in global scope or within a basic process
            boolean globalScope;
            if (nameType.equals(NameType.GlobalConstant)) {
                globalScope = true;
            } else {
                globalScope = false;
            }
            
            // horizontal definition
            code = this.horizDefCode(declName.toString(), circusType, expr, globalScope);
            
        } else if (box.equals(Box.SchBox)) {
            // do nothing, schemas are not translated unless they are the state, in which
            // case they are translated within basic process
            code = "";
        }
        
        return code;
    }
       
    /**
     * This returns the code for an axiomatic definition. Two templates are accepted:
     * 
     * 1) Variable has a RefExpr as a type, equality in the predicate part.
     *
     *      \begin{axdef}
     *          limit: \nat
     *      \where
     *          limit = 30
     *      \end{axdef}
     *
     * which will be translated to a method without parameters:
     *
     *      <modifiers>* CircusInteger limit() {
     *
     *          CircusInteger r = null;
     *          r = new CircusInteger(30);
     *          return r;
     *      }
     *
     * 2) Variable has a function type, conjunction of equalities in the predicate part.
     *
     *      \begin{axdef}
     *          birthday: NAME \pfun \nat
     *      \where
     *          birthday John = 1104
     *          birthday Mary = 2510
     *          birthday Jenny = 0703
     *      \end{axdef}
     *
     * which will be translated to a method with one parameter:
     *
     *      <modifiers>* CircusInteger birthday (NAME p) {
     * 
     *          CircusInteger r = null;
     *          if (p.equals(new NAME(NAME.John)))
     *              r = new CircusInteger(1104);
     *          if (p.equals(new NAME(NAME.Mary)))
     *              r = new CircusInteger(2510);
     *          if (p.equals(new NAME(NAME.Jenny)))
     *              r = new CircusInteger(0703);
     *          return r;
     *      }
     *
     * <modifiers>*     The modifiers depdends on where the axiomatic definition is declared:
     *
     * public static        Global scope
     * private              Within a basic process
     *
     * @param   CircusType  circusType      Type of the variable declared in the axiomatic definition
     * @param   Pred        pred            Predicate part of axiomatic definition
     * @param   String      methodName      Variable declared in the axiomatic definition, which will name the method 
     * @param   boolean     globalScope     true if the ax. def. is declared in the global scope; false, otherwise  
     *
     * @return  String      Code for a method, which is the translation of an axiomatic definition
     *
     * @see     axDefPredicateCode, horizDefCode
     */
    private String axDefCode(String methodName, CircusType circusType, Pred pred, boolean globalScope) {
        
        String code;
        String returnType = null;
        String paramType = null;
        
        Expr typeExpr = circusType.getExpression();
        
        if (typeExpr instanceof RefExpr) {
            
            boolean mixfix = ((RefExpr) typeExpr).getMixfix();
            List exprs = ((RefExpr) typeExpr).getExpr();
            
            if (mixfix) {
                // Generic operator application, ex: Color \pfun \nat
                // TODO: Fix this GAMBI when integrating with type-checker
                
                if (exprs.size() == 2) {
                    
                    Expr paramExpr = (Expr) exprs.get(0);
                    Expr returnExpr = (Expr) exprs.get(1);

                    if (paramExpr instanceof RefExpr && returnExpr instanceof RefExpr) {
                        
                        paramType = ((RefExpr) paramExpr).getRefName().toString();
                        returnType = ((RefExpr) returnExpr).getRefName().toString();
                        
                        // GAMBI !!!!
                        if (paramType.equals(ZString.NAT)) {
                            paramType = "CircusInteger";
                        }
                        
                        if (returnType.equals(ZString.NAT)) {
                            returnType = "CircusInteger";
                        }
                        
                    } else {/*!(paramExpr instanceof RefExpr && returnExpr instanceof RefExpr)*/
                        // "Use names in expressions used for declarations."
                        reportError(new Error(ErrorMessage.AXDEF_FORMAT));
                        System.out.println("AXDEF 1!");
                        //return "*************************";
                    }   
                    
                } else { /*!(exprs.size() == 2)*/
                    // "Format of function is not allowed.
                    reportError(new Error(ErrorMessage.AXDEF_FORMAT));
                    System.out.println("AXDEF 2!");
                    //return "*************************";
                }
                
            } else {
                // Name, ex: \nat
                returnType = circusType.getJavaCircusTypeName();
            }
            
        } else if (typeExpr instanceof PowerExpr) {
            
            Expr expr = ((PowerExpr) typeExpr).getExpr();
            
            if (expr instanceof RefName) {

                // Set, ex: \power \nat 
                returnType = "SetOfType";
                
            } else if (expr instanceof ProdExpr) {
            
                // Function, ex: \power (Name \cross \nat)
                List exprs = ((ProdExpr) expr).getExpr();
                if (exprs.size() == 2) {
                    
                    Expr paramTypeExpr = (Expr) exprs.get(0);
                    Expr returnTypeExpr = (Expr) exprs.get(1);
                    
                    if (paramTypeExpr instanceof RefExpr && returnTypeExpr instanceof RefExpr) {
                        
                        CircusType circusTypeParam = Util.getCircusTypeAnn(((RefExpr) paramTypeExpr).getRefName());
                        CircusType circusTypeReturn = Util.getCircusTypeAnn(((RefExpr) returnTypeExpr).getRefName());
                                
                        paramType = circusTypeParam.getJavaCircusTypeName();
                        returnType = circusTypeReturn.getJavaCircusTypeName(); 
                        
                    } else {/*!(paramTypeExpr instanceof RefExpr && returnTypeExpr instanceof RefExpr)*/
                        reportError(new Error(ErrorMessage.AXDEF_FORMAT));
                        // "Invalid format of function in the axiomatic definition."
                        System.out.println("AXDEF 3!");
                        //return "*************************";
                    }
                } else { /*!(exprs.size() == 2)*/
                    reportError(new Error(ErrorMessage.AXDEF_FORMAT));
                    // "Format of function is not allowed."
                    System.out.println("AXDEF 4!");
                    //return "*************************";
                }
            } else { /*!(expr instanceof ProdExpr)*/
                reportError(new Error(ErrorMessage.AXDEF_FORMAT));
                // "Type of variable in axiomatic definition must be a reference or a function."
                System.out.println("AXDEF 5!");
                //return "*************************";
            }
        } else { /*!(expr instanceof PowerExpr)*/
            reportError(new Error(ErrorMessage.AXDEF_FORMAT));
            // "Type of variable in axiomatic definition must be a reference or a function."
            System.out.println("AXDEF 6!");
            //return "*************************";
        }
        String parameters = "";
        String modifiers = "";
        
        if (paramType != null)
            parameters = paramType + " p";

        if (globalScope)
            modifiers = "public static ";
        else
            modifiers = "private ";
        
        code = modifiers + returnType + " " + methodName + "(" + parameters + ") {\n";
        
        code = code + returnType + " r = null;\n";
        
        code = code + this.axDefPredicateCode(pred);
        
        code = code + "return r;\n";
        
        code = code + "}";
        
        return code;
    }
    
    /**
     * Returns the code for the predicate part of an axiomatic definition.
     *
     * Predicate part must be an equality or a conjunction of
     * equalities.
     *
     * @param   Pred    pred    Predicate part
     *
     * @return  String  Code
     *
     * @see axDefCode
     */
    private String axDefPredicateCode(Pred pred) {
        
        String code = "";
        
        if (pred instanceof AndPred) {
            
            // Conjunction
            Pred rightPred = ((AndPred) pred).getRightPred();
            Pred leftPred = ((AndPred) pred).getLeftPred();
            
            // Recursive call
            code = code + this.axDefPredicateCode(rightPred);
            code = code + this.axDefPredicateCode(leftPred);
            
        } else if (pred instanceof MemPred) {
            
            // Base case: equality
            Expr leftExpr = ((MemPred) pred).getLeftExpr();
            Expr rightExpr = ((MemPred) pred).getRightExpr();
            boolean mixfix = ((MemPred) pred).getMixfix();
            
            if (mixfix = false || !(rightExpr instanceof SetExpr))
                // Not an equality
                reportError(new Error(ErrorMessage.AXDEF_FORMAT));
                // "Predicate in axiomatic definition must be an equality or 
                // conjunction of equalities."

            // Expression in the right-hand side of the equality
            Expr expr = (Expr) ((SetExpr) rightExpr).getExpr().get(0);
            
            if (leftExpr instanceof RefExpr) {

                // Simple name, ex: limit = 30
                
                // Name
                code = "r = " + expr.accept(this) + ";\n";
                
            } else  if (leftExpr instanceof ApplExpr) {
                
                // Function application, ex: birthday John = 1104
                Expr leftExpr2 = ((ApplExpr) leftExpr).getLeftExpr(); // birthday
                Expr rightExpr2 = ((ApplExpr) leftExpr).getRightExpr(); // John
                
                if (leftExpr2 instanceof RefExpr && rightExpr2 instanceof RefExpr) {
                    
                    String functionName = ((RefExpr) leftExpr2).getRefName().toString();
                    String paramName = ((RefExpr) rightExpr2).getRefName().toString();
                    
                    // Gets the CircusType annotation for the parameter
                    CircusType circusType = Util.getCircusTypeAnn(((RefExpr)rightExpr2).getRefName());
                    String type = circusType.getJavaCircusTypeName();
                            
                    // Code for function
                    code = "if (p.equals(new " + type + "(" + type + "." + paramName + ")))\n";
                    code = code + "r = " + expr.accept(this) + ";\n";
                    
                } else /*!(leftExpr2 instanceof RefExpr && rightExpr2 instanceof RefExpr)*/
                    reportError(new Error(ErrorMessage.AXDEF_FORMAT));
                    // "Use only references."
                
            } else /*!(leftExpr instanceof ApplExpr)*/
                reportError(new Error(ErrorMessage.AXDEF_FORMAT));
                // "Right-hand side of an equality in an axiomatic definition 
                //  must be a reference or an application."
            
        } else /*!(pred instanceof MemPred)*/
            reportError(new Error(ErrorMessage.AXDEF_FORMAT));
            // "Predicate in axiomatic definition must 
            // be an equality or conjunction of equalities."
        
        return code;
    }
    
    /**
     * This returns the code for a horizontal definition.
     * 
     * 1) Variable has a RefExpr as a type, equality in the predicate part.
     *
     *      \begin{zed}
     *          Xposition == 2
     *      \end{zed}
     *
     * which will be translated to a method without parameters:
     *
     *      <modifiers>* CircusInteger Xposition() {
     *
     *          CircusInteger r = null;
     *          r = new CircusInteger(2);
     *          return r;
     *      }
     *
     * ATTENTION:
     * This is different from the translation strategy (see page 169). There, a horizontal definition
     * is regarded as another type. Therefore, constants declared in an axiomatic definition will always 
     * have set type, and will be declared in terms of a set extension and the set operators (\cup, \cap
     * and \setminus).
     *
     *
     * <modifiers>*     The modifiers depdends on where the axiomatic definition is declared:
     *
     * public static        Global scope
     * private              Within a basic process
     *
     * @param   CircusType  circusType      Type of the variable declared in the horizontal definition
     * @param   Expr        expr            Expression
     * @param   String      methodName      Variable declared in the horizontal definition, which will name the method 
     * @param   boolean     globalScope     true if the ax. def. is declared in the global scope; false, otherwise  
     *
     * @return  String      Code for a method, which is the translation of an axiomatic definition
     *
     * @see     axDefCode
     */
    private String horizDefCode(String methodName, CircusType circusType, 
            Expr expr, boolean globalScope) {

        String code;
        String modifiers;
        
        if (globalScope)
            modifiers = "public static ";
        else
            modifiers = "private ";

        String returnType = circusType.getJavaCircusTypeName();
        
        code = modifiers + returnType + " " + methodName + "() {\n";
        
        code = code + returnType + " r = null;\n";
        
        code = code + " r = " + expr.accept(this);
        
        code = code + "return r;\n";
        
        code = code + "}";
        
        return code;
    }
    
    /**
     * 
     * Main method of this class.
     *
     */
    public Object visitProcessPara(ProcessPara processPara) { 
        
        String code;

        DeclName declName = processPara.getDeclName();
        CircusProcess circusProcess = processPara.getCircusProcess();
        
        try {

            // Sets the name of the current process
            currentProcess_ = declName;

            code = this.attributesAndConstructor(processPara);
            code = code + "\npublic void run() {";
            code = code + (String) circusProcess.accept(this) + ".run();";
            code = code + "\n}";

            currentProcess_ = null;
            
        } catch (Throwable t) {
            throw new JCircusException("Exception in process " + declName, t);
        }
        
        return code;
        
    }
    
    /**
     *
     */
    public Object visitActionPara(ActionPara actionPara) { 
        
        String code = null;
        
        DeclName declName = actionPara.getDeclName();
        CircusAction circusAction = actionPara.getCircusAction();
        
        currentAction_ = declName;
        
        // Resets the index for parallelism and recursion
        this.index_ = -1;

        String parameters = "";

        if (circusAction instanceof ActionD) {

            if (circusAction instanceof ParamAction) {

                // Parameterised action
                List decls = ((ActionD) circusAction).getDecl();
                parameters = this.parameters(decls);
                circusAction = ((ActionD) circusAction).getCircusAction();

            } else if (circusAction instanceof ActionIte) {

                // IteratedAction - not parameterized
                parameters = "";

            } else
                throw new InvalidSubTypeException(circusAction.getClass());
        }

        code = "private void " + declName.toString() + "(" +
                parameters + ") {" + (String) circusAction.accept(this)  + //";" + 
                "\n}";
            
        currentAction_ = null;
        
        return code;
    }
    
    /** **************************************************************************************************************
     * Process Tree.
     * ***************************************************************************************************************
     */
    
    /**
     * 
     */
    public Object visitBasicProcess(BasicProcess basicProcess) { 
        
        String code = null;
        
        RefName stateSchemaRefName = basicProcess.getStateSchema();
        CircusAction mainAction = basicProcess.getMainAction();
        ListTerm actionParas = basicProcess.getCircusActionPara();
        //ListTerm nameSets = basicProcess.getCircusNameSetPara();
        ListTerm zParas = basicProcess.getZPara();
        
        VelocityContext velocityContext;
        Template template = null;
        
        String stateDeclCode = "";
        String zParaCode;
        String actionsCode;
        String mainActionCode;
        
        // Z paragraphs
        zParaCode = "";
        for (int i = 0; i < zParas.size(); i++) {
            Para para = (Para) zParas.get(i);
            
            if (para instanceof AxPara) {
                // Only axiomatic and horizontal definitions are translated
                zParaCode = zParaCode + para.accept(this);
            }
        }
        
        // Action paragraphs
        actionsCode = "";
        for (int i = 0; i < actionParas.size(); i++) {
            ActionPara actionPara = (ActionPara) actionParas.get(i);
            
            CircusAction circusAction = actionPara.getCircusAction();
            actionsCode = actionsCode + actionPara.accept(this);   
        }
        
        // State
        // stateSchemaRefName must be annotated with its type: Signature
        if (stateSchemaRefName != null) {
        
            CircusType circusType = (CircusType) stateSchemaRefName.getAnn(CircusType.class);
            
            // This must be a signature type
            Signature signature = ((SignatureExpression) circusType.getExpression()).getSignature();
            
            // OBS: Verify if the type checker annotates stateSchemaRefName with SignatureAnn....
            // Signature signature = Util.getSignatureAnn(stateSchemaRefName);
         
            stateDeclCode = this.visitingSignature(signature);
        } else {
            stateDeclCode = "";
        }
        
        // Code for the main action
        mainActionCode = (String) mainAction.accept(this);
        
        try {
            // Velocity code
            velocityContext = new VelocityContext();
            Velocity.init();
            velocityContext.put("stateDecl", stateDeclCode);
            velocityContext.put("zParas", zParaCode);
            velocityContext.put("actionParas", actionsCode);
            velocityContext.put("mainAction", mainActionCode);
            
            template = Velocity.getTemplate(Constants.TMP_BASICPROC);
            
            StringWriter stringWriter = new StringWriter();
            
            template.merge(velocityContext, stringWriter);
            stringWriter.close();
            code = stringWriter.toString();
            
        } catch (Exception e) {
            throw new VelocityException("Velocity exception for basic process", e);
        }
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitCallProcess(CallProcess callProcess) {
        
        String code = null;
        
        RefName refName = callProcess.getRefName();
        CallType callType = callProcess.getCallType();
        
        Integer procId = Util.getIdCircusProcessAnn(callProcess).getId();  
        
        if (callType == CallType.Normal) {
            
            /* Normal call to a process */
            code = "(new CSProcess() { public void run() {new " + refName.toString() + "(" +
                    this.extractChans(refName.toString(), procId, null) +
                    ").run(); } })";
            
        } else if (callType == CallType.Param) {
            
            /* Process call with params */
            ListTerm params = callProcess.getExpr();
            
            code = "(new CSProcess() { public void run() {new " + refName.toString() + "(" +
                    this.extractChans(refName.toString(), procId, params) +
                    ").run(); } })";
            
            /**
             * All the code for ParametizedProcess is no longer being used, since
             * there are no more parameterised process defined on-the-fly.
             */
            
        } else if (callType == CallType.Gen) {
            
            /* Instantiation of Generic Process */
            ListTerm exprs = callProcess.getExpr();
            
            //TODO: Code for this.
            
        } else if (callType == CallType.Index) {
            
            /* Instantiation of Indexed Process */
            ListTerm exprs = callProcess.getExpr();
            
            //TODO: Code for this.
            
        } else {
            throw new InvalidParameterException(
                    "Invalid Parameter (" + callType.getClass() + "): " + callType.toString());
        }
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitHideProcess(HideProcess hideProcess) {
        
        
        String code = null;
        
        CircusProcess circusProcess = hideProcess.getCircusProcess();
        
        code = (String) circusProcess.accept(this);
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitRenameProcess(RenameProcess renameProcess) {
        
        String code = null;
        
        CircusProcess circusProcess = renameProcess.getCircusProcess();
        ListTerm/*<Name>*/ oldNames = renameProcess.getOldNames();
        ListTerm/*<Name>*/ newNames = renameProcess.getNewNames();
        
        code = (String) circusProcess.accept(this);
        
        code = Util.renameVars(code, newNames, oldNames);
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitIntChoiceProcess(IntChoiceProcess intChoiceProcess) { 
        
        String code = null;
        
        List processes = this.getProcesses(intChoiceProcess, Constants.OP_INTCHOICE);
        CircusProcess circusProcess;
        
        code = "(new CSProcess() { public void run() {";
        code = code + "int choosen = RandomGenerator.generateNumber(1, " + 
                processes.size() + "); ";
        code = code + "switch(choosen) { ";
        
        for (int i = 0; i < processes.size(); i++) {
            circusProcess = (CircusProcess) processes.get(i);
            code = code + "case " + (i + 1) + ": {" + 
                    (String) circusProcess.accept(this) + ".run();" + " } break; ";
        }
        
        code = code + "} ";
        code = code + "} })";
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitParProcess(ParProcess parProcess) { 
        
        String code = null;
        
        CircusProcess leftProc =  parProcess.getLeftProc();
        CircusProcess rightProc =  parProcess.getRightProc();
        
        code = "(new CSProcess() { public void run() {";
        code = code + "new Parallel( new CSProcess[] { "
                + (String) leftProc.accept(this) + ", " 
                + (String) rightProc.accept(this) + "}).run();";
        code = code + "} })";
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitSeqProcess(SeqProcess seqProcess) {
        
        String code = null;
        
        CircusProcess leftProc = seqProcess.getLeftProc();
        CircusProcess rightProc = seqProcess.getRightProc();
        
        code = "(new CSProcess() { public void run() {";
        code = code + (String) leftProc.accept(this) + ".run(); ";
        code = code + (String) rightProc.accept(this) + ".run(); ";
        code = code + "} })";
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitParamProcess(ParamProcess paramProcess) {
        
        String code;
        
        //ListTerm decls = paramProcess.getDecl();
        CircusProcess circusProcess = paramProcess.getCircusProcess();
        
        // This will only work when the parametized process is the outermost 
        // in a process definition, because de declaration of the parameters 
        // will have been treated already: they will be attributes in the 
        // process class.
        // OK: see note I wrote at the end of pg. 160
        code = (String) circusProcess.accept(this);
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitIntChoiceProcessIte(IntChoiceProcessIte intChoiceProcessIte) {
        
        String code;
        
        ListTerm decls = intChoiceProcessIte.getDecl();
        CircusProcess circusProcess = intChoiceProcessIte.getCircusProcess();
        
        this.index_++;
        int index = this.index_;
        
        String codeProcessClass = "";
        
        VelocityContext velocityContext;
        Template template = null;
        
        String lAttributes = this.paramsDecl(decls);
        String lParamDeclConstructor = this.paramsArgs(decls);
        String lConstructorBody = this.assignParams(decls);
        String lMethodRun = (String) circusProcess.accept(this) + ".run();";
        String indexString = index + "";
        
        try {
            // Velocity code
            velocityContext = new VelocityContext();
            Velocity.init();
            
            velocityContext.put("lAttributes", lAttributes);
            velocityContext.put("lParamDeclConstructor", lParamDeclConstructor);
            velocityContext.put("lConstructorBody", lConstructorBody);
            velocityContext.put("lMethodRun", lMethodRun);
            velocityContext.put("index", indexString);
            velocityContext.put("runRecursion", "");
            
            template = Velocity.getTemplate(Constants.TMP_RECACTION);
            
            StringWriter stringWriter = new StringWriter();
            template.merge(velocityContext, stringWriter);
            stringWriter.close();
            codeProcessClass = stringWriter.toString();
            
        } catch (Exception e) {
            throw new VelocityException("Velocity exception for iterated internal choice", e);
        }
        
        code = "(new CSProcess() { public void run() {";
        
        // ChooseIndexVars
        List formatDecls = this.formatDeclarations(decls);
        VarDecl varDecl;
        DeclName name;
        CircusType circusType;
        String parameters = "";
        
        for (int i = 0; i < formatDecls.size(); i++) {
            
            varDecl = (VarDecl) formatDecls.get(i);
            name = (DeclName) varDecl.getDeclName().get(0);
            circusType = (CircusType) name.getAnn(CircusType.class);
            
            code = code + "\n" + circusType.getJavaCircusTypeName() + " " + 
                    name + " = new " + circusType.getJavaCircusTypeName() +
                    "(RandomGenerator.generateNumber(" + 
                    circusType.getJavaCircusTypeName() + ".MIN_VALUE, " +
                    circusType.getJavaCircusTypeName() + ".MAX_VALUE" + "));";
            
            if (i == 0)
                parameters = parameters + name;
            else
                parameters = parameters + ", " + name;
            
        }
        
        code = code + codeProcessClass;
        
        code = code + "new I_" + index + "(" + parameters + ").run(); } })";
        
        return code;
        
    }
    
    /**
     * 
     */
    public Object visitParProcessIte(ParProcessIte parProcessIte) {
        
        String code = null;
        this.index_++;
        
        String procVecName = "procVec_" + this.index_;
        
        try {
            code = "(new CSProcess() { public void run() {";
            code = code + this.instProcesses(procVecName, parProcessIte, this.index_);
            
            code = code + "\nCSProcess[] processes_" + this.index_ + " = new CSProcess[" + procVecName + ".size()];";
            code = code + "\nfor (int i = 0; i < " + procVecName + ".size(); i++) {";
            code = code + "\nprocesses_" + this.index_ + "[i] = (CSProcess)" + procVecName + ".get(i);";
            code = code + "}\n (new Parallel(processes_" + this.index_ + ")).run(); } })";
            
        } catch (Exception e) {
            throw new VelocityException("Velocity Exception for iterated parallel process", e);
        }
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitSeqProcessIte(SeqProcessIte seqProcessIte) { 
        
        String code = null;
        this.index_++;
        
        String procVecName = "procVec_" + this.index_;
        
        try {
            code = "(new CSProcess() { public void run() {";
            code = code + this.instProcesses(procVecName, seqProcessIte, this.index_);
            
            code = code + "\nfor (int i = 0; i < " + procVecName + ".size(); i++) {";
            code = code + "\n((CSProcess)" + procVecName + ".get(i)).run();";
            code = code + "} } })";
            
        } catch (Exception e) {
            throw new VelocityException("Velocity Exception for iterated sequential process", e);
        }
        
        return code;
    }
    
    /** **************************************************************************************************************
     * CircusAction Tree.
     * ***************************************************************************************************************
     */
    
    /**
     * 
     */
    public Object visitSkipAction(SkipAction skipAction) {
        
        String code;
        
        code = "(new Skip()).run();";
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitStopAction(StopAction stopAction) {
        
        String code;
        
        code = "(new Stop()).run();";
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitChaosAction(ChaosAction chaosAction) {
        
        String code;
        
        code = "while(true){} ";
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitGuardedAction(GuardedAction guardedAction) {
        
        String code = "";
        
        Pred pred = guardedAction.getPred();
        CircusAction circusAction = guardedAction.getCircusAction();
        
        code += "if(" + pred.accept(this) + ") {";
        
        code += (String) circusAction.accept(this);
        
        code += "} else { (new Stop()).run(); }";
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitMuAction(MuAction muAction) {
        
        String code = null;
        
        DeclName declName = muAction.getDeclName();
        CircusAction circusAction = muAction.getCircusAction();
        
        VelocityContext velocityContext;
        Template template = null;
        
        this.index_++;
        int index = this.index_;
        
        // Sets the recursive name
        this.currentMuActionName_ = declName.toString();
        
        // Gets the local environment
        NameTypeEnv localEnv = (NameTypeEnv) muAction.getAnn(NameTypeEnv.class);
        
        List/*<String>*/ oldVariables = new ArrayList();
        List/*<String>*/ newVariablesLeft = new ArrayList();
        getVarsSubstitution(localEnv, index, oldVariables, newVariablesLeft, null);
                
        String lAttributes = this.attributesRecursion(localEnv, index);
        String lParamDeclConstructor = this.constructorParametersRecursion(localEnv);
        String lConstructorBody = this.constructorBodyRecursion(localEnv, index);
        
        String lMethodRun = this.substituteRecursion((String) circusAction.accept(this) 
                + ";", declName.toString(), localEnv, index);
        lMethodRun = Util.renameVars(lMethodRun, newVariablesLeft, oldVariables);
        
        String indexString = index + "";
        String runRecursion = this.runRecursion(index, index, localEnv);

        try {
            
            velocityContext = new VelocityContext();
            Velocity.init();
            
            velocityContext.put("lAttributes", lAttributes);
            velocityContext.put("lParamDeclConstructor", lParamDeclConstructor);
            velocityContext.put("lConstructorBody", lConstructorBody);
            velocityContext.put("lMethodRun", lMethodRun);
            velocityContext.put("index", indexString);
            velocityContext.put("runRecursion", runRecursion);
            
            template = Velocity.getTemplate(Constants.TMP_RECACTION);
            
            StringWriter stringWriter = new StringWriter();
            template.merge(velocityContext, stringWriter);
            stringWriter.close();
            code = stringWriter.toString();
            
        } catch (Exception e) {
            throw new VelocityException("Velocity Exception in recursive action", e);
        }
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitCallAction(CallAction callAction) {
        
        String code = "";
        
        CallType callType = callAction.getCallType();
        RefName refName = callAction.getRefName();
        
        if (callType == CallType.Normal) {
            
            /* Normal call to a process */
            
            if (this.currentMuActionName_ != null && 
                    this.currentMuActionName_.equals(refName.toString())) {
                // Recursive call
                code = refName.toString() + " ";
            } else {
                // Normal call
                code = refName.toString() + "();";
            }
            
        } else if (callType == CallType.Param) {
            
            /* Process call with params */
            ListTerm params = callAction.getExpr();
            Expr expr;
            
            code = refName.toString() + "(";
            
            for (int i = 0; i < params.size(); i++) {
                expr = (Expr) params.get(i);
                
                if (i > 0) {
                    code = code + ", ";
                }
                code = code + (String) expr.accept(this);
            }
            
            code = code + ");";
            
        } else {
            throw new InvalidParameterException("Invalid Parameter (" + 
                    callType.getClass() + "): " + callType.toString());
        }
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitSeqActionIte(SeqActionIte seqActionIte) {
        
        String code = "";
        
        ListTerm decls = seqActionIte.getDecl();
        CircusAction circusAction = seqActionIte.getCircusAction();
        
        Expr expression;
        
        VelocityContext velocityContext;
        Template template = null;
        
        this.index_++;
        int index = this.index_;
        
        NameTypeEnv localEnv = (NameTypeEnv) seqActionIte.getAnn(NameTypeEnv.class);
        
        List/*<String>*/ oldVariables = new ArrayList();
        List/*<String>*/ newVariablesLeft = new ArrayList();
        getVarsSubstitution(localEnv, index, oldVariables, newVariablesLeft, null);
        
        String indexString = index + "";
        String lAttributes = this.attributesInstantiation(decls, localEnv, index);
        String lParamDeclConstructor = this.constructorParametersInstantiation(decls, localEnv);
        String lConstructorBody = this.constructorBodyInstantiation(decls, localEnv, index);
        String lMethodRun = Util.renameVars((String) circusAction.accept(this), newVariablesLeft, oldVariables);
        String runIteratedSequentialAction = this.runIteratedSequentialAction(decls, index, localEnv);
        
        try {
            // Velocity code
            velocityContext = new VelocityContext();
            Velocity.init();
            
            velocityContext.put("index", indexString);
            velocityContext.put("lAttributes", lAttributes);
            velocityContext.put("lParamDeclConstructor", lParamDeclConstructor);
            velocityContext.put("lConstructorBody", lConstructorBody);
            velocityContext.put("lMethodRun", lMethodRun);
            velocityContext.put("runRecursion", runIteratedSequentialAction);
            
            // This uses the same template as recursive action
            template = Velocity.getTemplate(Constants.TMP_RECACTION);
            
            StringWriter stringWriter = new StringWriter();
            template.merge(velocityContext, stringWriter);
            stringWriter.close();
            code = stringWriter.toString();
            
        } catch (Exception e) {
            throw new VelocityException("Velocity Exception", e);
        }
        
        return code;
    }
    
    /**
     * Not ok?
     */
    public Object visitIntChoiceActionIte(IntChoiceActionIte intChoiceActionIte) {
        
        String code = null;
        
        ListTerm decls = intChoiceActionIte.getDecl();
        CircusAction circusAction = intChoiceActionIte.getCircusAction();
        Expr expression;
        
        VelocityContext velocityContext;
        Template template = null;
        
        this.index_++;
        int index = this.index_;
        
        NameTypeEnv localEnv = (NameTypeEnv) intChoiceActionIte.getAnn(NameTypeEnv.class);
        
        List/*<String>*/ oldVariables = new ArrayList();
        List/*<String>*/ newVariablesLeft = new ArrayList();
        getVarsSubstitution(localEnv, index, oldVariables, newVariablesLeft, null);
        
        String indexString = index + "";
        String lAttributes = this.attributesInstantiation(decls, localEnv, index);
        String lParamDeclConstructor = this.constructorParametersInstantiation(decls, localEnv);
        String lConstructorBody = this.constructorBodyInstantiation(decls, localEnv, index);
        String lMethodRun = Util.renameVars((String) circusAction.accept(this), newVariablesLeft, oldVariables);
        String runIteratedInternalChoiceAction = this.runIteratedInternalChoiceAction(decls, index, localEnv);
        
        try {
            
            velocityContext = new VelocityContext();
            Velocity.init();
            
            velocityContext.put("index", indexString);
            velocityContext.put("lAttributes", lAttributes);
            velocityContext.put("lParamDeclConstructor", lParamDeclConstructor);
            velocityContext.put("lConstructorBody", lConstructorBody);
            velocityContext.put("lMethodRun", lMethodRun);
            velocityContext.put("runRecursion", runIteratedInternalChoiceAction);
            
            // This uses the same template as recursive action
            template = Velocity.getTemplate(Constants.TMP_RECACTION);
            
            StringWriter stringWriter = new StringWriter();
            template.merge(velocityContext, stringWriter);
            stringWriter.close();
            code = stringWriter.toString();
            
        } catch (Exception e) {
            throw new VelocityException("Velocity Exception", e);
        }
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitExtChoiceAction(ExtChoiceAction extChoiceAction) {
        
        String code = "";
        
        //CircusAction leftAction = extChoiceAction.getLeftAction();
        //CircusAction rightAction = extChoiceAction.getRightAction();
        
        List<CircusAction> actions = Util.getActions(extChoiceAction, Constants.OP_EXTCHOICE);
        // These arrays have the same size as "actions"
        List<String> initialChannels = this.getInitialChannelsStr(actions);
        List<Pred> guards = this.getGuards(actions);

        CircusAction circusAction;
        String channelName;
                
/*
        final int CONST_CH = 0;
        final int CONST_CH2 = 1;

        Object[] sync;
        Vector seqOfSync = new Vector();
        Vector seqOfNotSync = new Vector();

        if (ch.isMultiSync()) {
            sync = new Object[] { 
                ch.getFromControlerId(), 
                ch.getChannel(),
                new Integer(ch.getProcessId()),
                new Integer(0)
            };
            seqOfSync.addElement(sync);
        } else {
            seqOfNotSync.addElement(ch.getChannel());
        }

        if (ch2.isMultiSync()) {
            sync = new Object[] { 
                ch2.getFromControlerId(), 
                ch2.getChannel(),
                new Integer(ch2.getProcessId()),
                new Integer(0)
            };
            seqOfSync.addElement(sync);
        } else {
            seqOfNotSync.addElement(ch2.getChannel());
        }

        MultiSyncClient client = new MultiSyncClient(seqOfSync,seqOfNotSync, null);
        client.run();

        switch(client.getChoosen()) {
            case CONST_CH:
                client.getValueTrans();
                (new Skip()).run();
                break;
            case CONST_CH2:
                client.getValueTrans();
                (new Skip()).run();
                break;
        }

 **/        

        String declConst = "";
        String init = "";
        String caseSwitch = "";
        String declGuards = "";
        String comma = "";
        
        for (int i = 0; i < initialChannels.size(); i++) {
            
            channelName = (String)initialChannels.get(i);
            circusAction = (CircusAction) actions.get(i);
            
            declConst += "final int " + this.extChoiceConst(channelName) + " = " + i + ";";
            init += this.extChoiceInit(channelName);
            caseSwitch += this.extChoiceCaseSwitch(channelName, circusAction);            
            declGuards += comma + (String) (guards.get(i)).accept(this);
            
            comma = ", ";
        }

        code += declConst;
        
	code += "Object[] sync;";
	code += "Vector seqOfSync = new Vector();";
	code += "Vector seqOfNotSync = new Vector();";
        
        code += init;
        
        code += "boolean[] g = {" + declGuards + "};";
        
        code += "MultiSyncClient client = new MultiSyncClient(seqOfSync, seqOfNotSync, null, g);";
        code += "client.run();";
        code += "switch(client.getChoosen()) {";
        code += caseSwitch;
        code += "}";
        
        return code;
    }
    
    /**
     *
     */
    private String extChoiceInit(String channelName) {
        
        String code = "";
        
        code += "if (" + channelName + ".isMultiSync()) {";
        code += "sync = new Object[] { ";
        code += channelName + ".getFromControlerId(), ";
        code += channelName + ".getChannel(),";
        code += "new Integer(" + channelName + ".getProcessId()),";
        code += "new Integer(0)";
        code += "};";
        code += "seqOfSync.addElement(sync);";
        code += "} else {";
        code += "seqOfNotSync.addElement(" + channelName + ".getChannel());";
        code += "}";
        
/*
            if (ch.isMultiSync()) {
                sync = new Object[] { 
                    ch.getFromControlerId(), 
                    ch.getChannel(),
                    new Integer(ch.getProcessId()),
                    new Integer(0)
                };
                seqOfSync.addElement(sync);
            } else {
                seqOfNotSync.addElement(ch.getChannel());
            }
 */        
        return code;
    }
    
    /**
     * circusAction is an action of one of the following types: 
     * 
     * GuardedAction 
     *      |
     *      +-- PrefixingAction
     *
     * or 
     *
     * PrefixingAction
     *
     */
    private String extChoiceCaseSwitch(String channelName, CircusAction circusAction) {
        
        String code = "";     
        
        CircusAction actionWithoutPrefixing = this.prefixingActionWithoutGuard(circusAction);
        
        code += "case " + this.extChoiceConst(channelName) + ": "; 
        this.externalChoice_ = true;
        
        if (actionWithoutPrefixing != null) {
            code += (String) actionWithoutPrefixing.accept(this);
        }
        code += "break;";
        
        return code;        
    }
    
    /**
     *
     */
    private CircusAction prefixingActionWithoutGuard(CircusAction circusAction) {
        
        CircusAction result = null;
        
        if (circusAction instanceof GuardedAction) {
            result = this.prefixingActionWithoutGuard(((GuardedAction)circusAction).getCircusAction());
        } else if (circusAction instanceof PrefixingAction) {
            result = circusAction;
        } else {
            Object[] params = new Object[] { 
                    currentProcess_.toString(), currentAction_.toString()};
            reportError(new Error(ErrorMessage.EXTCH_BRANCH, params));
        }
        
        return result;
    }

    private String extChoiceConst(String channelName) {
        return "CONST_" + channelName.toUpperCase();
    }
    
    /**
     *
     */
    public Object visitIntChoiceAction(IntChoiceAction intChoiceAction) {
        
        String code;
        
        CircusAction leftAction = intChoiceAction.getLeftAction();
        CircusAction rightAction = intChoiceAction.getRightAction();
        
        List actions = Util.getActions(intChoiceAction, Constants.OP_INTCHOICE);
        CircusAction circusAction;
        
        code = "\nint choosen = RandomGenerator.generateNumber(1, " + actions.size() + ");";
        code = code + "\nswitch (choosen) {";
        
        for (int i = 0; i < actions.size(); i++) {
            circusAction = (CircusAction) actions.get(i);
            code = code + "\ncase " + i + ": { " + (String) circusAction.accept(this) + "} break;";
        }
        
        code = code + "\n}";
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitSeqAction(SeqAction seqAction) {
        
        String code;
        
        CircusAction leftAction = seqAction.getLeftAction();
        CircusAction rightAction = seqAction.getRightAction();
        
        code = (String) leftAction.accept(this) + ";\n" +  (String) rightAction.accept(this);
        
        return code;
    }
    
    /**
     * Not OK ?
     */
    public Object visitParAction(ParAction parAction) {
        
        String code = null;
        VelocityContext velocityContext;
        Template template = null;
        
        CircusAction leftAction = parAction.getLeftAction();
        CircusAction rightAction = parAction.getRightAction();
        
        NameSet fstVariables = parAction.getLeftNameSet();
        NameSet sndVariables = parAction.getRightNameSet();
        
        this.index_++;
        int index = this.index_;
        
        List/*<String>*/ fstNames = this.getNames(fstVariables);
        List/*<String>*/ sndNames = this.getNames(sndVariables);
        NameTypeEnv localEnv = (NameTypeEnv) parAction.getAnn(NameTypeEnv.class);
        
        List/*<String>*/ oldVariables = new ArrayList();
        List/*<String>*/ newVariablesLeft = new ArrayList();
        List/*<String>*/ newVariablesRight = new ArrayList();
        getVarsSubstitution(localEnv, index, oldVariables, newVariablesLeft, 
                newVariablesRight);
        
        String lName = "ParallelLeftBranch_" + index;
        String lAttributes = this.attributesParallel(fstNames, localEnv, index, TranslatorVisitor.LEFT);
        String lParamDeclConstructor = this.constructorParametersParallel(fstNames, localEnv);
        String lConstructorBody = this.constructorBodyParallel(fstNames, localEnv, index, TranslatorVisitor.LEFT);
        
        String lMethodRun = Util.renameVars((String)leftAction.accept(this) + ";", newVariablesLeft, oldVariables);
        
        String lParameters = this.parametersParallel(fstNames, localEnv);
        String rName = "ParallelRightBranch_" + index;
        String rAttributes = this.attributesParallel(sndNames, localEnv, index, TranslatorVisitor.RIGHT);
        String rParamDeclConstructor = this.constructorParametersParallel(sndNames, localEnv);
        String rConstructorBody = this.constructorBodyParallel(sndNames, localEnv, index, TranslatorVisitor.RIGHT);
        
        String rMethodRun = Util.renameVars((String) rightAction.accept(this) + ";", newVariablesRight, oldVariables);
        
        String rParameters = this.parametersParallel(sndNames, localEnv);
        String indexString = index + "";
        String mergeVars = this.mergeVars(fstNames, localEnv, lName, index, TranslatorVisitor.LEFT);
        mergeVars = mergeVars + this.mergeVars(sndNames, localEnv, rName, index, TranslatorVisitor.RIGHT);
        
        try {
            velocityContext = new VelocityContext();
            Velocity.init();
            
            velocityContext.put("lName", lName);
            velocityContext.put("lAttributes", lAttributes);
            velocityContext.put("lParamDeclConstructor", lParamDeclConstructor);
            velocityContext.put("lConstructorBody", lConstructorBody);
            velocityContext.put("lMethodRun", lMethodRun);
            velocityContext.put("lParameters", lParameters);
            velocityContext.put("rName", rName);
            velocityContext.put("rAttributes", rAttributes);
            velocityContext.put("rParamDeclConstructor", rParamDeclConstructor);
            velocityContext.put("rConstructorBody", rConstructorBody);
            velocityContext.put("rMethodRun", rMethodRun);
            velocityContext.put("rParameters", rParameters);
            velocityContext.put("index", indexString);
            velocityContext.put("mergeVars", mergeVars);
            
            template = Velocity.getTemplate(Constants.TMP_ACTPARALLEL);
            
            StringWriter stringWriter = new StringWriter();
            template.merge(velocityContext, stringWriter);
            stringWriter.close();
            code = stringWriter.toString();
            
        } catch (Exception e) {
            throw new VelocityException("Velocity exception", e);
        }
        
        return code;
    }

    /*
     * Important: This is a report of a problem in the old parser.
     *
        public Object visit (ApplicationExpression applicationExpression) {
 
                String code;
 
                Expression first = applicationExpression.getFirst();
                Expression second = applicationExpression.getSecond();
 
                // This should be a Parameterized Action
                // TODO: Parser 1.
 
                // When this is a parameterized action, 'first' will be a RefName and
                // 'second' will be either a ParenthesedExpression (one param.) or a
                // TupleExtension (more than one param.)
                code = first.toString() + "(" + this.javaExpression(second) + ");";
                //System.out.println("Application Expression");
 
                return code;
        }*/
    
    /**
     * 
     */
    public Object visitPrefixingAction(PrefixingAction prefixingAction) {
        
        String code = "";
        ChanUse chanUse = null;
        
        Communication communication = prefixingAction.getCommunication();
        CircusAction circusAction = prefixingAction.getCircusAction();
        
        RefName chanName = communication.getChanName();
        ListTerm actuals = communication.getGenActuals();
        ListTerm fields = communication.getChanFields();
        
        if (fields.size() >= 1) {
            
            // Synchronization on multiple values
            
            // Instantiation of the array
            String arrayCode = "";
            
            // instantiation of the generic channel
            if (actuals != null && actuals.size() > 0) {
                for (int i = 0; i < actuals.size(); i++) {
                    
                    // Expression used for instantiation of a generic channel
                    Expr expr = (Expr) actuals.get(i);
                    String typeName = "";
                    
                    /**
                     * In the future, when more types will be considered  
                     * this code must be changed in
                     * order to be more flexible and support the other types.
                     */
                    if (expr instanceof RefExpr) {
                        // Free type or CircusInteger
                        typeName = ((RefExpr) expr).getRefName().toString();
                          
                        if (typeName.equals(MathToolkitConstants.NAT)) {
                            // Circus Integer
                            typeName = "CircusInteger";
                        }
                    } else { /*!(expr instanceof RefExpr)*/
                        Object[] params = new Object [] { 
                            currentProcess_, currentAction_, 
                            MathToolkitConstants.NAT,
                            MathToolkitConstants.NUM };
                        reportError(new Error(ErrorMessage.CHAN_INST, params));
                    }
                    
                    arrayCode = arrayCode + "[Type." + typeName + "]";
                }
            }
            
            // instantiation of the parameters
            for (int i = 0; i <= fields.size() - 2; i++) {
                
                Field field = (Field) fields.get(i);
                
                if (!(field instanceof DotField)) {
                    Object[] params = new Object[] { 
                        currentProcess_, currentAction_, chanName };
                    reportError(new Error(ErrorMessage.CHAN_FORMAT, params));
                }
                
                Expr expr = ((DotField) field).getExpression();
                
                arrayCode = arrayCode + "[ (" + (String) expr.accept(this) + ").getValue() ]";
            }
            
            // Gets the last parameter
            Field field = (Field) fields.get(fields.size() - 1);
            
            if (field instanceof InputField) {
                
                RefName variable = ((InputField) field).getVariable();
                CircusType circusType = (CircusType) variable.getAnn(CircusType.class);
                
                if (this.externalChoice_) {
                    // In the case this is an external choice, the value has already been read.
                    // Get the transfered value in the client
                    code = "\n{ " + circusType.getJavaCircusTypeName() + " " 
                                + variable + " = (" + circusType.getJavaCircusTypeName()
                                + ") client.getValueTrans(); " 
                                + (String) circusAction.accept(this) 
                                + "\n}";

                    // Resets the flag
                    this.externalChoice_ = false;
                    
                } else {
                    code = "\n{ " + circusType.getJavaCircusTypeName() + " " 
                                + variable + " = (" + circusType.getJavaCircusTypeName()
                                + ") " + chanName.toString() + arrayCode + ".read();\n" 
                                + (String) circusAction.accept(this) 
                                + "\n}";
                }
                
            } else if (field instanceof OutputField) {
                
                Expr expr = ((OutputField) field).getExpression();
                
                code = "\n" + chanName.toString() + arrayCode + ".write(" + 
                        (String) expr.accept(this) + ");\n" + 
                        (String) circusAction.accept(this);
                
            } else if (field instanceof DotField) {
                
                // TODO: How to treat this case in the external choice ????
                
                Expr expr = ((DotField) field).getExpression();
                
                code = "\n" + chanName.toString() + arrayCode + "[ (" + 
                        (String) expr.accept(this) + ").getValue() ]" + 
                        ".communicate(null);\n" + (String) circusAction.accept(this);
                
            } else
                throw new InvalidSubTypeException(field.getClass());
            
        } else {
            
            // No fields: Synchronization channel
            
            ProcChanEnv processChannelEnvironment =
                    this.environment_.getProcessChannelEnvironment(this.currentProcess_.toString());

            try {
                chanUse = processChannelEnvironment.getJavaChannelType(chanName.toString());
            } catch (ChannelNotDefinedExceptionForProcess e) {
                Object[] params = new Object[] { 
                        currentProcess_, currentAction_, chanName };
                reportError(new Error(ErrorMessage.CHAN_NOT_DEF, params));
            }

            if (chanUse.equals(ChanUse.Input) || chanUse.equals(ChanUse.AltInput)) {
                // In the case this is an external choice, the value has already been read
                if (!this.externalChoice_)
                    code = "\n" + chanName.toString() + ".read(); ";
                
                this.externalChoice_ = false;
                
            } else if (chanUse.equals(ChanUse.Output)) {
                code = "\n" + chanName.toString() + ".write(null); ";
            } else if (chanUse.equals(ChanUse.Undefined)) {
                code = "\n" + chanName.toString() + ".communicate(null); ";
            } else 
                throw new InvalidParameterException(chanUse.toString());

            code = code + (String) circusAction.accept(this);
                
        }
        
        return code;
    }
    
    
    /** **************************************************************************************************************
     * Command Tree.
     * ***************************************************************************************************************
     */
    
    /**
     * 
     */
    public Object visitAssignmentCommand(AssignmentCommand assignmentCommand) {
        
        String code = "";
        
        /**
         * This is result of a change in the AST. No more assignment pairs 
         * (see Manuela's email 13/08/2005
         * - Circus.xsd e CircusTools.xsd)
         */
        //List assignmentPairs = assignmentCommand.getAssignmentPair();
        
        List leftVars = assignmentCommand.getLeftVars();
        List exprs = assignmentCommand.getExprs();
        
        //AssignmentPair assignPair;
        RefName refName;
        Expr expr;
        CircusType circusType;
        
        // Declaration of the aux variables
/*        for (int i = 0; i < assignmentPairs.size(); i++) {
            
            assignPair = (AssignmentPair) assignmentPairs.get(i);
            refName = assignPair.getLHS();
            expr = assignPair.getRHS();
            
            circusType = (CircusType) refName.getAnn(CircusType.class);
            
            code = code + "\n" + circusType.getJavaCircusTypeName() + 
                " aux_" + refName.toString() + " = " + (String) expr.accept(this) + ";";
        }*/
        
        // Assignment of the aux_variables to the original variables
        //for (int i = 0; i < assignmentPairs.size(); i++) {
        for (int i = 0; i < leftVars.size(); i++) {
            
            //assignPair = (AssignmentPair) assignmentPairs.get(i);
            refName = (RefName) leftVars.get(i); //assignPair.getLHS();
            expr = (Expr) exprs.get(i); //assignPair.getRHS();
            
            code = code + "\n" + refName.toString() + " = " + 
                    (String) expr.accept(this) + ";"; //+ "aux_" + refName.toString() + ";";
        }
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitIfGuardedCommand(IfGuardedCommand ifGuardedCommand) {
       
        String code = "";
        
        List guardedActions = ifGuardedCommand.getGuardedAction();
        
        ProcChanEnv procChanEnv = null;
        GuardedAction guardedAction;
        Pred predicate;
        Action action;
        
        for (int i = 0; i < guardedActions.size(); i++) {
            
            guardedAction = (GuardedAction) guardedActions.get(i);
            
            Pred pred = guardedAction.getPred();
            CircusAction circusAction = guardedAction.getCircusAction();
            
            if (i != 0) {
                code = code + "\n} else ";
            } else {
                code = "\n";
            }
            code = code + "if (" + (String) pred.accept(this) + ") {\n" + 
                    circusAction.accept(this) + ";";
        }
        
        code = code + "} else { while(true){} }";
        
        return code;
    }
    
    /**
     * 
     */
    public Object visitVarDeclCommand(VarDeclCommand varDeclCommand) {
        
        String code;
        
        List decls = varDeclCommand.getDeclarations();
        CircusAction circusAction = varDeclCommand.getCircusAction();
        
        Decl decl;
        
        code = "\n{";
        
        for (int i = 0; i < decls.size(); i++) {
        
            decl = (Decl) decls.get(i);
            
            this.isPrivate_ = false;
            code = code + decl.accept(this);
        }
        
        code = code + (String) circusAction.accept(this) + ";";
        
        code = code + "\n}";
        
        return code;
    }
    
    /** **************************************************************************************************************
     * Decl Tree.
     * ***************************************************************************************************************
     */
    
    /**
     * OK
     */
    public Object visitVarDecl(VarDecl varDecl) {
        
        String code = "\n";
        
        ListTerm names = varDecl.getDeclName();
        Expr expression = varDecl.getExpr();
        
        // Get the Java type in the annotation of one of the variables
        // TODO: this is a gambi - check how CZT does this
        CircusType circusType = (CircusType) ((DeclName) names.get(0)).getAnn(CircusType.class);
        
        // Checks in the global state if this is private or not
        if (this.isPrivate_) {
            code = "\nprivate ";
        }
        
        code = code + circusType.getJavaCircusTypeName() + " ";
        
        for (int i = 0; i < names.size(); i++) {
            if (i != 0) {
                code = code + ", ";
            }
            code = code + ((DeclName) names.get(i)).toString() + " = null";
        }
        
        code = code + ";";
        
        return code;
    }
    
    /** **************************************************************************************************************
     * Pred Tree.
     * ***************************************************************************************************************
     */
    
    public Object visitFalsePred(FalsePred fact) {
        
        return "false";
    }
    
    public Object visitTruePred(TruePred fact) {
        
        return "true";
    }
    
    public Object visitNegPred(NegPred negPred) {
        
        String code;
        Pred pred = negPred.getPred();
        
        code = "(!(" + pred.accept(this) + "))";
        
        return code;
    }
    
    public Object visitAndPred(AndPred andPred) {
        
        String code;
        
        Pred leftPred = andPred.getLeftPred();
        Pred rightPred = andPred.getRightPred();
        
        code = "(" + leftPred.accept(this) + " && " + rightPred.accept(this) + ")";
        
        return code;
    }
    
    public Object visitOrPred(OrPred orPred) {
        
        String code;
        
        Pred leftPred = orPred.getLeftPred();
        Pred rightPred = orPred.getRightPred();
        
        code = "(" + leftPred.accept(this) + " || " + rightPred.accept(this) + ")";
        
        return code;
    }
    
    public Object visitIffPred(IffPred iffPred) {
        
        String code;
        
        Pred leftPred = iffPred.getLeftPred();
        Pred rightPred = iffPred.getRightPred();
        
        code = "(" + leftPred.accept(this) + " == " + rightPred.accept(this) + ")";
        
        return code;
    }
    
    public Object visitImpliesPred(ImpliesPred impliesPred) {
        
        String code;
        
        Pred leftPred = impliesPred.getLeftPred();
        Pred rightPred = impliesPred.getRightPred();
        
        code = "(!(" + leftPred.accept(this) + ") || " + rightPred.accept(this) + ")";
        
        return code;
    }
    
    public Object visitExprPred(ExprPred exprPred) {
        
        String code;
        
        Expr expr = exprPred.getExpr();
        
        code = (String) expr.accept(this);
        
        return code;        
    }
    
    public Object visitMemPred(MemPred memPred) {
        
        /**
         * A relation operator application (C.5.12). The mixfix attribute is false
         * iff the input has the form Expr1 \in Expr2.
         *
         * When mixfix = true:
         *
         * - the second (right) Expr might be a relational operator and the
         * first (left) Expr must be a tuple containing the corresponding number of
         * arguments (unless the operator has one argument, then no tuple is required)
         *
         * - OR the expression represents an equality, and the second (right) Expr
         * must be a set expression containing exactly one expression.
         *
         * For example, the input "m < n" generates a MemPred element with mixfix=true,
         * left=(m,n) and right=" _ < _ ", whereas "(m,n) \in (_ < _)" has the same
         * left and right expressions, but mixfix=false.
         *
         * The input "m = n" generates a MemPred element with mixfix=true, left=m and
         * right={n}.
         *
         */
        String code = "ERROR";
        
        Expr leftExpr = memPred.getLeftExpr();
        Expr rightExpr = memPred.getRightExpr();
        Boolean mixfix = memPred.getMixfix();
        
        if (mixfix == Boolean.TRUE) {
            
            if (rightExpr instanceof SetExpr) {
                // Equality
                
                // Gets the only element in the set
                Expr expr = (Expr) ((SetExpr) rightExpr).getExpr().get(0);
                
                code = "((" + (String) leftExpr.accept(this) + ").getValue() == (" +
                        (String) expr.accept(this) + ").getValue())";
                
            } else {
                // Relational Operator Application
                code = visitingMathToolkit((RefExpr) rightExpr, leftExpr);
            }
        } else {
            // Expr1 \in Expr 2
                    Object[] params = new Object[] { 
                        currentProcess_, currentAction_};
                    //reportError(new Error(ErrorMessage.EXPR_MEMB, params));
                    return "************************";
        }
        
        return code;
    }
    
    /**
     *
     *  A function application (C.6.21, C.6.22). 
     *
     *  C.6.21 (Function Operator Application). For example: S + T. In this case, Mixfix=true, 
     *  the first (left) expression is the name, (" _ + _ "), (that is, a RefExpr with Mixfix=false!) 
     *  and the second (right) expression is (S,T). 
     *
     *  C.6.22 (Application). For example: dom R. In this case, Mixfix=false, the first (left) 
     *  expression is the function, dom, (again, a RefExpr with Mixfix=false) and the second (right) 
     *  expression is the argument R. 
     *
     *  As another example, m + n has Mixfix=true, whereas (_ + _)(m, n) has Mixfix=false. 
     *
     *
     */
    public Object visitApplExpr(ApplExpr applExpr) {
        
        String code = "ERROR";
        
        Expr leftExpr = applExpr.getLeftExpr();
        Expr rightExpr = applExpr.getRightExpr();
        Boolean mixfix = applExpr.getMixfix();
        
        if (mixfix == Boolean.TRUE) {
            // Function Operator Application
            code = visitingMathToolkit((RefExpr) leftExpr, rightExpr);
            
        } else {
            
            // Function Application
            if (leftExpr instanceof RefExpr && rightExpr instanceof RefExpr) {
                
                code = ((RefExpr) leftExpr).getRefName() + "(" + 
                        ((RefExpr) rightExpr).getRefName() + ")";
                        
            } else {
                Object[] params = new Object[] { 
                        currentProcess_, currentAction_ };
                reportError(new Error(ErrorMessage.FUNCAPP_FORMAT, params));
            }
        }
        
        return code;
    }    
    
    
    /** **************************************************************************************************************
     * Expr Tree.
     * ***************************************************************************************************************
     */
    
    /**
     *
     */
    public Object visitRefExpr(RefExpr refExpr) {
        
        String code = "*ERRO*";
        
        RefName refName = refExpr.getRefName();
        
        // Gets the NameType annotation
        NameType nameType = (NameType) refName.getAnn(NameType.class);
        
        if (nameType.equals( NameType.StateComponent)) {
            code = refName.toString();
            
        } else if (nameType.equals( NameType.ProcessParam)) {
            code = refName.toString();
            
        } else if (nameType.equals( NameType.ActionParam)) {
            code = refName.toString();
            
        } else if (nameType.equals( NameType.LocalVariable)) {
            // local variable or process parameter or action parameter
            code = refName.toString();
            
        } else if (nameType.equals( NameType.Channel)) {
            // This should not happen
            
        } else if (nameType.equals( NameType.GlobalConstant)) {
            // abbreviation or axDef declared in the global scope
            code = "AxiomaticDefinitions." + refName.toString() + "()";
            
        } else if (nameType.equals( NameType.LocalConstant)) {
            code = refName.toString() + "()";
            
        } else if (nameType.equals( NameType.FreeType)) {
            // TODO: implement this
            code = "TODO";
            
        } else if (nameType.equals( NameType.ElemFreeType)) {
            DeclName declName = new Factory().createDeclName(refName.toString());
            String nameFreeType = this.environment_.getNameFreeType(declName);
            code = "new " + nameFreeType + "(" + nameFreeType + "." + refName.toString() + ")";
            
        } else if (nameType.equals( NameType.ProcessName)) {
            throw new InvalidParameterException("The type of the reference expression " 
                    + refName.toString() + " is " + nameType);
            
        } else if (nameType.equals( NameType.ActionName)) {
            throw new InvalidParameterException("The type of the reference expression " 
                    + refName.toString() + " is " + nameType);
            
        } else if (nameType.equals( NameType.SchemaName)) {
            throw new InvalidParameterException("The type of the reference expression " 
                    + refName.toString() + " is " + nameType);
            
        } else if (nameType.equals( NameType.RecursiveName)) {
            throw new InvalidParameterException("The type of the reference expression " 
                    + refName.toString() + " is " + nameType);
            
        } else if (nameType.equals( NameType.ChannelSetName)) {
            throw new InvalidParameterException("The type of the reference expression " 
                    + refName.toString() + " is " + nameType);
        }
        
        return code;
    }
    
    /**
     *
     */
    public Object visitNumExpr(NumExpr numExpr) {
        
        String code;
        BigInteger bigInteger = numExpr.getValue();
        
        code = "new CircusInteger(" +  bigInteger.toString() + ")";
        
        return code;
    }
    
    /**
     *
     */
    public Object visitSetExpr(SetExpr setExpr) {
        
        String code;
        
        List exprs = setExpr.getExpr();
        
        code = "new Set(new Object[] {";
        
        for (int i = 0; i < exprs.size(); i++) {
            
            Expr expr = (Expr) exprs.get(i);
            code = code + expr.accept(this);
        }
        
        code = code + "});";
        
        return code;
    }
    
    /** **************************************************************************************************************
     * Mathematical Toolkit.
     * ***************************************************************************************************************
     */
    
    /**
     *
     */
    private String visitingMathToolkit(RefExpr opName, Expr expr) {
        
        String code = "ERROR";
        /**
         * - the second (right) Expr must be either a relational operator and the
         * first (left) Expr must be a tuple containing the corresponding number of
         * arguments (unless the operator has one argument, then no tuple is required)
         *
         */
        
        // Name of the operator
        String op = opName.getRefName().toString();
        
        // Expressions in case it is a n-ary operator, n > 2.
        ListTerm exprs = null;
        Expr expr1 = null, expr2 = null;
        
        if (expr instanceof TupleExpr) {
            exprs = ((TupleExpr) expr).getExpr();
            
            if (exprs.size() == 2) {
                // Binary operation
                expr1 = (Expr) exprs.get(0);
                expr2 = (Expr) exprs.get(1);
            }
        }
        
        // TODO: Do the rest of the mathematical toolkit
        if (op.equals(MathToolkitConstants.ADD)) {
            code = (String) expr1.accept(this) + ".add(" + (String) expr2.accept(this) + ")";
        } else if (op.equals(MathToolkitConstants.SUB)) {
            code = (String) expr1.accept(this) + ".sub(" + (String) expr2.accept(this) + ")";
        } else if (op.equals(MathToolkitConstants.MULT)) {
            code = (String) expr1.accept(this) + ".mult(" + (String) expr2.accept(this) + ")";
        } else if (op.equals(MathToolkitConstants.DIV)) {
            code = (String) expr1.accept(this) + ".div(" + (String) expr2.accept(this) + ")";
        } else if (op.equals(MathToolkitConstants.MOD)) {
            code = (String) expr1.accept(this) + ".mod(" + (String) expr2.accept(this) + ")";
        } else if (op.equals(MathToolkitConstants.NEGATE)) {
            code = (String) expr.accept(this) + ".negate()";
        } else if (op.equals(MathToolkitConstants.LESS)) {
            code = (String) expr1.accept(this) + ".less(" + (String) expr2.accept(this) + ")";
        } else if (op.equals(MathToolkitConstants.LESS_EQUAL)) {
            code = (String) expr1.accept(this) + ".lessEqual(" + (String) expr2.accept(this) + ")";
        } else if (op.equals(MathToolkitConstants.GREATER)) {
            code = (String) expr1.accept(this) + ".greater(" + (String) expr2.accept(this) + ")";
        } else if (op.equals(MathToolkitConstants.GREATER_EQUAL)) {
            code = (String) expr1.accept(this) + ".greaterEqual(" + (String) expr2.accept(this) + ")";
        } else if (op.equals(MathToolkitConstants.NOTEQUAL)) {
            code = "!" + (String) expr1.accept(this) + ".equals(" + (String) expr2.accept(this) + ")";
        } else {
            Object[] params = new Object[] {
                    currentProcess_, currentAction_, op };
            //reportError(new Error(ErrorMessage.OP_UNSUPP, params));
            return "*************************";
        }
        return code;
    }
    
    /** **************************************************************************************************************
     * Util Methods - Extract Chans
     * ***************************************************************************************************************
     */

    /**
     * (This is called in visitCallProcess)
     */
    private String extractChans(String procName, Integer procId, List parameters/*, boolean isMain*/) {
        
        String code = "";
        
        ChanUseEnv channelUseEnvironment = this.environment_.getProcessChannelEnvironment(procName).getChanUseEnvVis();
        Iterator iteratorKeys = channelUseEnvironment.iteratorKeys();
        String comma = "";
        String channelName;
                
        Expr expr;
        
        while (iteratorKeys.hasNext()) {
            
            channelName = (String) iteratorKeys.next();
            code = code + comma + this.channelCode(channelName, procId, /*isMain*/ false);
            
            if (comma.equals(""))
                comma = ", ";
        }
        
        if (parameters != null) {
            for (int i = 0; i < parameters.size(); i++) {
                
                expr = (Expr) parameters.get(i);
                code = code + comma + expr.accept(this);
                
                if (comma.equals(""))
                    comma = ", ";
            }
        }
        
        return code;
    }
    
    /**
     *
     */
    private String channelCode(String channelName, Integer procId, boolean isMain) {
        
        if (isMain)
            return channelName; // This will never be called
        else
            //return "new GeneralChannel(" + channelName + ", \"" + processName + "\")"; 
            return "new GeneralChannel(" + channelName + ", new Integer(" + procId + "))"; 
    }       
    
    /** **************************************************************************************************************
     * Util Methods.
     * ***************************************************************************************************************
     */
    
    /**
     * Returns the java code that refers to the declaration of attributes and the
     * declaration of the constructor of a class, which results from the translation of a
     * Process Definition.
     *
     */
    private String attributesAndConstructor(ProcessPara processPara) {
        
        String code = "";
        List decls;
        
        CircusProcess circusProcess = processPara.getCircusProcess();
        DeclName declName = processPara.getDeclName();
        
        ProcChanEnv procChanEnv = this.environment_.getProcessChannelEnvironment(declName.toString());
        ChanUseEnv chanUseEnvVis = procChanEnv.getChanUseEnvVis();
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();
        ChanSyncEnv chanSyncEnv = procChanEnv.getChanSyncEnv();
        NameTypeEnv nameTypeEnv = procChanEnv.getNameTypeEnv();
    
        
        Integer idProc = Util.getIdCircusProcessAnn(
                circusProcess).getId();
        
        Iterator iteratorVisible = chanUseEnvVis.iteratorKeys();
        Iterator iteratorHidden = chanUseEnvHid.iteratorKeys();
        
        String channelName;
        ChanUse chanUse; 
        ChanSync syncType;
        CircusType circusTypeChannel;
        String brackets = "";
        
        // Attributes: visible channels
        while(iteratorVisible.hasNext()) {
            channelName = (String) iteratorVisible.next();
            chanUse = chanUseEnvVis.get(channelName);
            
            syncType = chanSyncEnv.get(channelName);
            circusTypeChannel = nameTypeEnv.getCircusType(channelName);
            
            brackets = Util.arrayDim(circusTypeChannel, syncType, 0);
            
            code = code + "\nprivate GeneralChannel " + brackets + " " + channelName + ";";
        }
        
        // hidden channels
        while(iteratorHidden.hasNext()) {
            channelName = (String) iteratorHidden.next();
            chanUse = chanUseEnvHid.get(channelName);
            syncType = chanSyncEnv.get(channelName);
            
            circusTypeChannel = nameTypeEnv.getCircusType(channelName);
            brackets = Util.arrayDim(circusTypeChannel, syncType, 0);
            
            code = code + "\nprivate GeneralChannel " + brackets + " " + channelName + ";";
        }
        
        // parameters of process
        if (circusProcess instanceof ParamProcess) { 
            
            decls = ((ParamProcess) circusProcess).getDecl();
            
            Decl decl;
            
            for (int i = 0; i < decls.size(); i++) {
                
                decl = (Decl) decls.get(i);
                
                this.isPrivate_ = true;
                code = code + decl.accept(this);
            }
        } else {
            decls = null;
        }
        
        // Constructor
        code = code +
                "\npublic " + declName.toString() + "(" + 
                this.constructorParameters(procChanEnv, decls) + ") {";
        
        code = code + this.constructorBody (processPara, decls);
        
        code = code + "\n}";
        
        return code;
    }
    
    
    /*
     * Aux method for Internal and External Choice Processes.
     */
    
    /**
     * This method return all the processes that take part in an internal or
     * external choice in the first level. Basically it transforms the binary
     * operation into an n-ary operation.
     *
     */
    private List getProcesses(CircusProcess process, int operation) {
        
        List r = new ArrayList();
        
        if ((operation == Constants.OP_EXTCHOICE && process instanceof ExtChoiceProcess)
        || (operation == Constants.OP_INTCHOICE && process instanceof IntChoiceProcess)) {
            
            r.addAll(getProcesses(((Process2) process).getLeftProc(), operation));
            r.addAll(getProcesses(((Process2) process).getRightProc(), operation));
            
        } else if (operation == Constants.OP_EXTCHOICE || operation == Constants.OP_INTCHOICE) {
            
            r.add(process);
            
        } else
            throw new InvalidParameterException("Operation not defined: " + operation);
        
        return r;
    }
    
    /**
     * Translates a List of declarations into parameters.
     *
     * @param declarations
     * @return
     */
    private String parameters(List decls) {
        return this.parameters(decls, true);
    }
    

    private String parameters(List decls, boolean isFirst) {
        
        String code = "";
        Decl decl;
        
        List names;
        CircusType circusType;
        
        if (decls != null) {
            for (int i = 0; i < decls.size(); i++) {
                decl = (Decl) decls.get(i);
                
                if (decl instanceof VarDecl) {
                    
                    names = ((VarDecl) decl).getDeclName();
                    circusType = (CircusType) ((TermA) names.get(0)).getAnn(CircusType.class);
                    
                    for (int j = 0; j < names.size(); j++) {
                        if (!isFirst)
                            code = code + ", ";
                        else
                            isFirst = false;
                        
                        code = code + circusType.getJavaCircusTypeName() + 
                                " " + names.get(j);
                    }
                }
            }
        }
        
        return code;
    }
    
    /**
     *
     */
    private String constructorParameters(ProcChanEnv processChannelEnvironment, List decls) {
        
        String code = "";
        boolean isFirst = true;
        
        ChanUseEnv channelUseEnvironmentVisible = processChannelEnvironment.getChanUseEnvVis();
        ChanSyncEnv channelSyncEnvironment = processChannelEnvironment.getChanSyncEnv();
        NameTypeEnv channelTypeEnvironment = processChannelEnvironment.getNameTypeEnv();
        
        Iterator iterator = channelUseEnvironmentVisible.iteratorKeys();
        String channelName;
        ChanUse chanUse;
        ChanSync syncType;
        CircusType circusTypeChannel;
        String brackets = "";
        
        while (iterator.hasNext()) {
            channelName = (String) iterator.next();
            chanUse = channelUseEnvironmentVisible.get(channelName);
            syncType = channelSyncEnvironment.get(channelName);
            
            circusTypeChannel = channelTypeEnvironment.getCircusType(channelName);
            brackets = Util.arrayDim(circusTypeChannel, syncType, 0);
            
            if (!isFirst)
                code = code + ", ";
            else
                isFirst = false;
            
            code = code + chanUse.toJava() + brackets + " " + channelName;
        }
        
        code = code + parameters(decls, isFirst);
        
        return code;
    }
    

    /**
     *
     */
    private String constructorBody(ProcessPara processPara, List decls) {
        
        String code = "";
        boolean first = true;
        
        CircusProcess circusProcess = processPara.getCircusProcess();
        Integer procId = Util.getIdCircusProcessAnn(circusProcess).getId();
        DeclName declName = processPara.getDeclName();
        
        ProcChanEnv procChanEnv = this.environment_.getProcessChannelEnvironment(declName.toString());
        ChanUseEnv chanUseEnvVis = procChanEnv.getChanUseEnvVis();
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();
        ChanSyncEnv chanSyncEnv = procChanEnv.getChanSyncEnv();
        NameTypeEnv nameTypeEnv = procChanEnv.getNameTypeEnv();
        ChanInfoEnv multiSyncEnv = procChanEnv.getChanInfoEnv();
        
        Iterator iteratorVisible = chanUseEnvVis.iteratorKeys();
        Iterator iteratorHidden = chanUseEnvHid.iteratorKeys();
        String channelName;
        int javaTypeChannel, syncType;
        Decl decl;
        List names;
        CircusType circusTypeChannel;
        int dim;
        ProcChanUseEnv chanMsEnv;
        
        // Visible channels
        while (iteratorVisible.hasNext()) {
            channelName = (String) iteratorVisible.next();
            code = code + "\nthis." + channelName + " = " + channelName + ";";
        }
        
        // Hidden channels
        while (iteratorHidden.hasNext()) {
            channelName = (String) iteratorHidden.next();
            //syncType = chanSyncEnv.getCircusType(channelName);
            circusTypeChannel = this.environment_.getCircusType(channelName);
            chanMsEnv = multiSyncEnv.get(channelName);
            
            // Use InstArray for communication
            code = code + Util.instChannelSimple(channelName, chanMsEnv, procId, false);
        }
        
        // Declarations
        if (decls != null) {
            for (int i = 0; i < decls.size(); i++) {
                decl = (Decl) decls.get(i);
                
                if (decl instanceof VarDecl) {
                    
                    names = ((VarDecl) decl).getDeclName();
                    
                    for (int j = 0; j < names.size(); j++) {
                        code = code + "\nthis." + names.get(j) + " = " + names.get(j) + ";";
                    }
                }
            }
        }
        
        return code;
    }
    
    /*
     * This method is called by action parallelism; it returns the code with the initialization
     * of local variables for one of the processes of the parallelism.
     *
     * @param   List    varsPartition           The variables of the partition for the process.
     * @param   TypeEnvironment localEnv     The local environment.
     * @param   int     index           The index used in the parallelism.
     * @param   String  leftRight       L or R, which indicates for which side of the parallelism this code is.
     */
    private String attributesParallel(List/*<String>*/ varsPartition, 
            NameTypeEnv localEnv, int index, String leftRight) {
        
        String code = "";
        Enumeration keys = localEnv.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            circusType = localEnv.getCircusType(declName.toString());
            
            if (nameType.equals(NameType.StateComponent) || 
                    nameType.equals(NameType.ProcessParam)) {

                /**
                 * State comp. and process param. are declared and initialized.
                 * They are class attributes and can be seen inside the inner class.
                 */
                code = code + "\npublic " + circusType.getJavaCircusTypeName() 
                        + " aux_" + leftRight + "_" + declName.toString() + "_" + index 
                        + " = " + declName.toString() + ";";
                
            } else if (nameType.equals(NameType.LocalVariable) ||
                    nameType.equals(NameType.ActionParam)) {
                
                /**
                 * Local variables and action parameters are declared and will
                 * be initialized in the constructor.
                 */
                code = code + "\npublic " + circusType.getJavaCircusTypeName() 
                        + " aux_" + leftRight + "_" + declName.toString() + "_" + index + ";";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String constructorParametersParallel(List names, NameTypeEnv typeEnvironment) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        String comma = "";
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                circusType = typeEnvironment.getCircusType(declName.toString());
                
                code = code + comma + circusType.getJavaCircusTypeName() + 
                        " " + declName.toString();
                
                if (comma.equals(""))
                    comma = ", ";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String constructorBodyParallel(List names, 
            NameTypeEnv typeEnvironment, int index, String leftRight) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                circusType = typeEnvironment.getCircusType(declName.toString());
                code = code + "\naux_" + leftRight + "_" + declName.toString() + 
                        "_" + index + " = " + declName.toString() + ";";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String parametersParallel(List names, NameTypeEnv typeEnvironment) {
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        String comma = "";
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                circusType = typeEnvironment.getCircusType(declName.toString());
                code = code + comma + declName.toString();
                
                if (comma.equals(""))
                    comma = ", ";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String mergeVars(List names, NameTypeEnv typeEnvironment, 
            String lrName, int index, String leftRight) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        String comma = "";
        String declName;
        
        int lf = 0;
        if (leftRight == TranslatorVisitor.RIGHT)
            lf = 1;
        
        for (int i=0; i<names.size(); i++) {
            
            declName = (String) names.get(i);
            code = code + "\n" + declName + " = " + "((" + lrName + ")processes_" 
                    + index + "[" + lf + "]).aux_" +
                    leftRight + "_" + declName + "_" + index + ";";
        }
        
        return code;
    }

    /**
     * Receives the local environment, an index and inserts elements in the lists:
     * - varList: <var> 
     * - auxVarLeft: aux_LEFT_<var>_<index>
     * - auxVarRight: aux_RIGHT_<var>_<index>
     *
     * for each <var> in the local environment, if it is a parallelism is true, or 
     * each local variable and action parameter in local environment, 
     * if it is not.
     *
     * It is a parallelism if auxVarRight is not null.
     *
     * USAGE RULE: 'varList', 'auxVarLeft' and 'auxVarRight' must be initialized 
     * with an empty list before the call of this method.
     */
    private void getVarsSubstitution(NameTypeEnv localEnv, int index, 
                List varList, List auxVarLeft, List auxVarRight) {

        Enumeration keys = localEnv.keys();
        DeclName declName;
        NameType nameType;
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            /**
             * If it is a parallelism (auxVarRight != null), all variables in 
             * localEnv will be renamed; otherwise, only local vars and action 
             * parameters will.
             */
            if (auxVarRight != null || 
                    (nameType.equals(NameType.LocalVariable) || nameType.equals(NameType.ActionParam))) {
            
                varList.add(declName.toString());
                auxVarLeft.add("aux_" + TranslatorVisitor.LEFT + "_" + 
                        declName.toString() + "_" + index); 
                if (auxVarRight != null)
                    auxVarRight.add("aux_" + TranslatorVisitor.RIGHT + "_" + 
                            declName.toString() + "_" + index); 
            }
        }
    }
    
    /**
     *
     */
    private List/*<String>*/ getOldVariables(NameTypeEnv typeEnvironment, 
            boolean isParallelism) {
        
        List r = new ArrayList();
        
        Enumeration keys = typeEnvironment.keys();
        DeclName declName;
        NameType nameType;
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam) ||
                    (nameType.equals( NameType.StateComponent) && isParallelism)) {
                
                r.add(declName.toString());
            }
        }
        return r;
    }
    
    /**
     *
     */
    private List/*<String>*/ getNewVariables(NameTypeEnv localEnv, 
            String leftRight, int index, boolean isParallelism) {
        
        List r = new ArrayList();
        
        Enumeration keys = localEnv.keys();
        DeclName declName;
        NameType nameType;
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            /**
             * If it is a parallelism, all variables in localEnv will be renamed.
             * If it is not, only local vars and action parameters will.
             */
            if (isParallelism || 
                    (nameType.equals(NameType.LocalVariable) || nameType.equals(NameType.ActionParam))) {
            
                r.add("aux_" + leftRight + "_" + declName.toString() + "_" + index);
            }
        }
        return r;
    }
    
    /*
     * Aux methods for recursion.
     */
    private String attributesRecursion(NameTypeEnv typeEnvironment, int index) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        
        DeclName declName;
        NameType nameType;
        CircusType circusType;
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                circusType = typeEnvironment.getCircusType(declName.toString());
                
                code = code + "\npublic " + circusType.getJavaCircusTypeName() + 
                        " aux_left_" + declName.toString() + "_" + index + ";";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String constructorParametersRecursion(NameTypeEnv typeEnvironment) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        String comma = "";
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                circusType = typeEnvironment.getCircusType(declName.toString());
                
                code = code + comma + circusType.getJavaCircusTypeName() + " " + declName.toString();
                
                if (comma.equals(""))
                    comma = ", ";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String constructorBodyRecursion(NameTypeEnv typeEnvironment, int index) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                circusType = typeEnvironment.getCircusType(declName.toString());
                code = code + "\naux_left_" + declName.toString() + "_" + 
                        index + " = " + declName.toString() + ";";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String parametersRecursion(NameTypeEnv typeEnvironment) {
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        String comma = "";
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                circusType = typeEnvironment.getCircusType(declName.toString());
                code = code + comma + declName.toString();
                
                if (comma.equals(""))
                    comma = ", ";
            }
        }
        return code;
    }
    
    /**
     *
     */    
    private String mergeVarsRecursion(NameTypeEnv typeEnvironment, 
            int index, int newIndex) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        DeclName declName;
        NameType nameType;
        
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                
                code = code + "\n" + declName.toString() + " = i_" + 
                        index + "_" + newIndex +
                        ".aux_left_" + declName.toString() + "_" + index + ";";
            }
        }
        
        return code;
    }
    
    /**
     *
     */    
    private String substituteRecursion(String initialCode, String recName, 
            NameTypeEnv typeEnvironment, int index) {
        
        String code = "";
        
        StringTokenizer st = new StringTokenizer(initialCode, ".;,(){} \t\n\r\f", true);
        String token, newToken;
        int newIndex = 0;
        
        while (st.hasMoreTokens()) {
            
            token = st.nextToken();
            
            if (recName.equals(token)) {
                
                token = this.runRecursion(index, newIndex, typeEnvironment);
                newIndex++;
            }
            
            code = code + token;
        }
        
        return code;
    }
    
    /**
     *
     */   
    private String runRecursion(int index, int newIndex, NameTypeEnv typeEnvironment) {
        
        String code = "";
        
        code = "\nI_" + index + " i_" + index + "_" + newIndex + " = new I_" + index + "(" +
                parametersRecursion(typeEnvironment) + "); ";
        code = code + "\ni_" + index + "_" + newIndex + ".run(); ";
        code = code + "\n" + this.mergeVarsRecursion(typeEnvironment, index, newIndex);
        
        return code;
    }
    
    /*
     * Aux methods for Instantiation of Action.
     */
    private String attributesInstantiation(List decls, 
            NameTypeEnv typeEnvironment, int index) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        
        Decl decl;
        List names;
        
        // Declarations
        for (int i = 0; i < decls.size(); i++) {
            decl = (Decl) decls.get(i);
            
            if (decl instanceof VarDecl) {
                
                names = ((VarDecl) decl).getDeclName();
                circusType = (CircusType) ((TermA) names.get(0))
                .getAnn(CircusType.class);
                
                for (int j = 0; j < names.size(); j++) {
                    
                    code = code + "\nprivate " + circusType.getJavaCircusTypeName() + " " + names.get(j) + ";";
                }
            }
        }
        
        // Local Env
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            circusType = typeEnvironment.getCircusType(declName.toString());
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                
                code = code + "\npublic " + circusType.getJavaCircusTypeName() + " aux_left_"
                        + declName.toString() + "_" + index + ";";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String constructorParametersInstantiation(List decls, NameTypeEnv typeEnvironment) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        String comma = "";
        
        Decl decl;
        List names;
        boolean isFirst = true;
        
        // Declarations
        for (int i = 0; i < decls.size(); i++) {
            decl = (Decl) decls.get(i);
            
            if (decl instanceof VarDecl) {
                
                names = ((VarDecl) decl).getDeclName();
                circusType = (CircusType) ((TermA) names.get(0)).getAnn(CircusType.class);
                
                for (int j = 0; j < names.size(); j++) {
                    
                    code = code + comma + circusType.getJavaCircusTypeName() + " " + names.get(j);
                    
                    if (comma.equals(""))
                        comma = ", ";
                    
                }
            }
        }
        
        // Local Env
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                circusType = typeEnvironment.getCircusType(declName.toString());
                
                code = code + comma + circusType.getJavaCircusTypeName() + " " + declName.toString();
                
                if (comma.equals(""))
                    comma = ", ";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String constructorBodyInstantiation(List decls, NameTypeEnv typeEnvironment, int index) {
        
        String code = "";
        Enumeration keys = typeEnvironment.keys();
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        
        Decl decl;
        List names;
        
        // Declarations
        for (int i = 0; i < decls.size(); i++) {
            decl = (Decl) decls.get(i);
            
            if (decl instanceof VarDecl) {
                
                names = ((VarDecl) decl).getDeclName();
                
                for (int j = 0; j < names.size(); j++) {
                    code = code + "\nthis." + names.get(j) + " = "
                            + names.get(j) + ";";
                }
            }
        }
        
        // Local Env
        while (keys.hasMoreElements()) {
            
            declName = (DeclName) keys.nextElement();
            nameType = (NameType) declName.getAnn(NameType.class);
            
            if (nameType.equals( NameType.LocalVariable) ||
                    nameType.equals( NameType.ActionParam)) {
                
                circusType = typeEnvironment.getCircusType(declName.toString());
                code = code + "\naux_left_" + declName.toString() + "_" + index + " = " + declName.toString() + ";";
            }
        }
        return code;
    }
    
    /**
     *
     */
    private String parametersInstantiation(List/*DeclName*/ parameters, 
            NameTypeEnv typeEnvironment) {
        
        String code = "";
        Enumeration keys;
        CircusType circusType;
        DeclName declName;
        NameType nameType;
        String comma = "";
        
        Expr expr;
        List names;
        
        boolean isFirst = true;
        
        // Declarations
        for (int i = 0; i < parameters.size(); i++) {
            //expr = (Expr) parameters.getCircusType(i);
            //code = code + comma + expr.accept(this);
            declName = (DeclName) parameters.get(i);
            code = code + comma + declName.toString();
            
            if (comma.equals(""))
                comma = ", ";
        }
        
        if (typeEnvironment != null) {
            
            keys = typeEnvironment.keys();
            // Local Env
            while (keys.hasMoreElements()) {
                
                declName = (DeclName) keys.nextElement();
                nameType = (NameType) declName.getAnn(NameType.class);
                
                if (nameType.equals( NameType.LocalVariable) ||
                        nameType.equals( NameType.ActionParam)) {
                    
                    circusType = typeEnvironment.getCircusType(declName.toString());
                    code = code + comma + declName.toString();
                    
                    if (comma.equals(""))
                        comma = ", ";
                }
            }
        }
        
        return code;
    }
    
    /**
     *
     */    
    private String parametersInstantiation(List/*DeclName*/ parameters) {
        return parametersInstantiation(parameters, null);
    }
    
    /**
     *
     */    
    private String runInstantiation(List parameters, int index, 
            int newIndex, NameTypeEnv typeEnvironment) {
        
        String code = "";
        
        code = "\nI_" + index + " i_" + index + "_" + newIndex + " = new I_" + index + "(" +
                this.parametersInstantiation(parameters, typeEnvironment) + "); ";
        code = code + "\ni_" + index + "_" + newIndex + ".run(); ";
        code = code + "\n" + this.mergeVarsRecursion(typeEnvironment, index, newIndex);
        
        return code;
    }
    
    /**
     *
     */    
    private List<String> getInitialChannelsStr(List actions) {
        
        List<String> result = new ArrayList();
        CircusAction circusAction;
        
        for (int i = 0; i < actions.size(); i++) {
            
            circusAction = (CircusAction) actions.get(i);
            result.add(getInitialChannel(circusAction));
        }
        
        return result;
    }
    
    private String getInitialChannel(CircusAction circusAction) {
        
        String code = "";
        
        // TODO: the rest of the types of CircusActions, including RefName (create an action environment,
        // to getCircusType the Action from its declName)
        if (circusAction instanceof GuardedAction) {
            code = getInitialChannel(((GuardedAction) circusAction).getCircusAction());
            
        } else if (circusAction instanceof PrefixingAction) {
            
            Communication communication = ((PrefixingAction) circusAction).getCommunication();
            code = communication.getChanName().toString();
        } else {
            Object[] params = new Object[] { 
                    currentProcess_, currentAction_ };
            reportError(new Error(ErrorMessage.EXTCH_BRANCH, params));
        }
        
        return code;
    }
    
    /**
     * Returns a List of String, representing the java code for the
     * predicates for the actions in an external choice.
     *
     * @param actions
     * @return
     */
    private List<Pred> getGuards(List<CircusAction> actions) {
        
        List<Pred> result = new ArrayList();
        CircusAction circusAction;
        
        for (int i = 0; i < actions.size(); i++) {
            
            circusAction = actions.get(i);
            result.add(this.getGuard(circusAction));
        }
        
        return result;
    }
    
    /**
     * Returns the guard of an action. If the action is not guarded, returns truth predicate.
     *
     */
    private Pred getGuard(CircusAction circusAction) {
        
        Pred result;
        
        if (circusAction instanceof GuardedAction) {
            
            Pred pred = ((GuardedAction) circusAction).getPred();
            CircusAction circusAction2 = ((GuardedAction) circusAction).getCircusAction();
            
            Pred pred2 = this.getGuard(circusAction2);
            
            if (pred2.equals(factory_.createTruePred()))
                // returns the guard
                result = pred;
            else
                // the child action is itself a guarded action, so return the 
                // conjunction of both guards
                result = factory_.createAndPred(pred, pred2, Op.And);
            
        } else {
            
            // Base case: an action that is not guarded has true as guard
            result = factory_.createTruePred();
        }
        
        return result;
        
    }
    
    /*
     * Aux methods for Iterated Sequential Action
     */
    private String runIteratedSequentialAction(List decls, int index, 
            NameTypeEnv typeEnvironment) {
        
        String code = "";
        
        List formDecls = formatDeclarations(decls);
        VarDecl varDecl;
        DeclName declName;
        Expr expression;
        CircusType circusType;
        String javaType;
        List parameters = new ArrayList();
        
        // Nested for's
        for (int i = 0; i < formDecls.size(); i++) {
            
            varDecl = (VarDecl) formDecls.get(i);
            declName = (DeclName) varDecl.getDeclName().get(0);
            expression = varDecl.getExpr();
            
            circusType = (CircusType) declName.getAnn(CircusType.class);
            javaType = circusType.getJavaCircusTypeName();
            
            parameters.add(declName);
            
            code = code + "\nfor (" + javaType + " " + declName + " = " + getValue(javaType, TranslatorVisitor.MIN) + "; ";
            code = code + declName + ".compareTo(" + getValue(javaType, TranslatorVisitor.MAX) + ") <= 0; ";
            code = code + declName + ".setValue(" + declName + ".getValue() + 1" + ")) {";
        }
        
        code = code + "I_" + index + " i_" + index + "_" + index + " =  new I_" + index + "(" +
                parametersInstantiation(parameters , typeEnvironment) + ");";
        code = code + "\ni_" + index + "_" + index + ".run();";
        code = code + mergeVarsRecursion(typeEnvironment, index, index);
        
        // Close nested for's
        for (int i = 0; i < formDecls.size(); i++) {
            code = code + "}";
        }
        
        return code;
    }
    
    /*
     * Aux methods for Iterated Internal Choice Action
     */
    private String runIteratedInternalChoiceAction(List decls, int index, 
            NameTypeEnv typeEnvironment) {
        
        String code = "";
        
        List formatDecls = formatDeclarations(decls);
        VarDecl varDecl;
        DeclName declName;
        CircusType circusType;
        List<DeclName> parameters = new ArrayList();
        
        for (int i = 0; i < formatDecls.size(); i++) {
            
            varDecl = (VarDecl) formatDecls.get(i);
            declName = (DeclName) varDecl.getDeclName().get(0);
            circusType = (CircusType) declName.getAnn(CircusType.class);
            
            code = code + "\n" + circusType.getJavaCircusTypeName() + " " + 
                    declName + " = new " + circusType.getJavaCircusTypeName() +
                    "(RandomGenerator.generateNumber(" + 
                    circusType.getJavaCircusTypeName() + ".MIN_VALUE, " +
                    circusType.getJavaCircusTypeName() + ".MAX_VALUE" + "));";
            
            parameters.add(declName);
        }

        code = code + "(new I_" + index + "(" + parametersInstantiation(parameters , typeEnvironment) + ")).run();";
        
        return code;
    }
    
    /**
     *
     */    
    private List formatDeclarations(List decls) {
        
        List r = new ArrayList();
        Decl decl;
        VarDecl newColonDecl;
        
        for (int i = 0; i < decls.size(); i++) {
            
            decl = (Decl) decls.get(i);
            
            if (decl instanceof VarDecl) {
                List names = ((VarDecl) decl).getDeclName();
                Expr expression = ((VarDecl) decl).getExpr();
                
                for (int j = 0; j < names.size(); j++) {
                    
                    List declNames = new ArrayList();
                    declNames.add(names.get(j));
                    newColonDecl = this.factory_.createVarDecl(declNames, expression);
                    r.add(newColonDecl);
                }
            } else
                throw new InvalidSubTypeException(decl.getClass());
        }
        
        return r;
    }
    
    /**
     *
     */
    private String getValue(String javaType, int type) {
        
        String code;
        String value = (type == TranslatorVisitor.MIN ? (value = "MIN_VALUE") : (value = "MAX_VALUE"));
        code = "new " + javaType + "(" + javaType + "." + value + ")";
        
        return code;
    }
    
    /**
     * Returns a List of String.
     */
    private List/*<String>*/ getNames(NameSet nameSet) {
        
        List r = new ArrayList();
        RefName refName;
        
        if (nameSet instanceof BasicNameSet) {
            
            List names = ((BasicNameSet) nameSet).getRefName();
            for (int i=0; i<names.size(); i++) {
                
                refName = (RefName) names.get(i);
                r.add(refName.toString());
            }
        }
        
        return r;
    }
    
    /**
     *
     */    
    private String paramsDecl(List decls) {
        
        String code = "";
        Decl decl;
        
        for (int i = 0; i < decls.size(); i++) {
            
            decl = (Decl) decls.get(i);
            
            this.isPrivate_ = true;
            code = code + (String) decl.accept(this);
        }
        
        return code;
    }
    
    /**
     *
     */    
    private String paramsArgs(List decls) {
        
        String code = "";
        CircusType circusType;
        String comma = "";
        
        Decl decl;
        List names;
        boolean isFirst = true;
        
        // Declarations
        for (int i = 0; i < decls.size(); i++) {
            decl = (Decl) decls.get(i);
            
            if (decl instanceof VarDecl) {
                
                names = ((VarDecl) decl).getDeclName();
                circusType = (CircusType) ((TermA) names.get(0)).getAnn(CircusType.class);
                
                for (int j = 0; j < names.size(); j++) {
                    
                    code = code + comma + circusType.getJavaCircusTypeName() + 
                            " " + names.get(j);
                    
                    if (comma.equals(""))
                        comma = ", ";
                    
                }
            }
        }
        
        return code;
    }
    
    /**
     * Receives a list of decls and, for each declName, returns:
     *
     * this.<name> = <name>;
     *
     * This method is used to form the body of a constructor.
     *
     * @param decls
     * @return
     */
    private String assignParams(List decls) {
        
        String code = "";
        CircusType circusType;
        
        Decl decl;
        List names;
        
        // Declarations
        for (int i = 0; i < decls.size(); i++) {
            decl = (Decl) decls.get(i);
            
            if (decl instanceof VarDecl) {
                
                names = ((VarDecl) decl).getDeclName();
                
                for (int j = 0; j < names.size(); j++) {
                    code = code + "\nthis." + names.get(j) + " = "
                            + names.get(j) + ";";
                }
            }
        }
        
        return code;
    }
    
    /**
     *
     */
    private String instProcesses(String procVecName, ProcessIte processIte, 
            int index) throws Exception {
        
        String code = "";
        
        VelocityContext velocityContext;
        Template template = null;
        
        List decls = processIte.getDecl();
        CircusProcess circusProcess = processIte.getCircusProcess();
        
        String lAttributes = paramsDecl(decls);
        String lParamDeclConstructor = this.paramsArgs(decls);
        String lConstructorBody = this.assignParams(decls);
        String lMethodRun = (String) circusProcess.accept(this) + ".run();";
        String indexString = index + "";
        
        velocityContext = new VelocityContext();
        Velocity.init();
        
        velocityContext.put("lAttributes", lAttributes);
        velocityContext.put("lParamDeclConstructor", lParamDeclConstructor);
        velocityContext.put("lConstructorBody", lConstructorBody);
        velocityContext.put("lMethodRun", lMethodRun);
        velocityContext.put("index", indexString);
        velocityContext.put("runRecursion", "");
        
        template = Velocity.getTemplate(Constants.TMP_RECACTION);
        
        StringWriter stringWriter = new StringWriter();
        template.merge(velocityContext, stringWriter);
        stringWriter.close();
        code = stringWriter.toString();
        
        code = "Vector " + procVecName + " = new Vector();\n" + code;
        
        List formDecls = formatDeclarations(decls);
        VarDecl varDecl;
        DeclName declName;
        Expr expression;
        CircusType circusType;
        String javaType;
        List parameters = new ArrayList();
        
        // Nested for's
        for (int i = 0; i < formDecls.size(); i++) {
            
            varDecl = (VarDecl) formDecls.get(i);
            declName = (DeclName) varDecl.getDeclName().get(0);
            expression = varDecl.getExpr();
            
            circusType = (CircusType) declName.getAnn(CircusType.class);
            javaType = circusType.getJavaCircusTypeName();
            
            parameters.add(declName);
            
            code = code + "\nfor (" + javaType + " " + declName + " = " + getValue(javaType, TranslatorVisitor.MIN) + "; ";
            code = code + declName + ".compareTo(" + getValue(javaType, TranslatorVisitor.MAX) + ") <= 0; ";
            code = code + declName + ".setValue(" + declName + ".getValue() + 1" + ")) {";
        }
        
        code = code + procVecName + ".add(new I_" + index + "(" + parametersInstantiation(parameters) + "));";
        
        // Close nested for's
        for (int i = 0; i < formDecls.size(); i++) {
            code = code + "} ";
        }
        
        return code;
    }
    
    /**
     * 
     */
    private String chooseIndexVars(List decls) throws Exception {
        
        String code = "";
        
        List formatDecls = formatDeclarations(decls);
        VarDecl varDecl;
        DeclName declName;
        CircusType circusType;
        String parameters = "";
        
        for (int i = 0; i < formatDecls.size(); i++) {
            
            varDecl = (VarDecl) formatDecls.get(i);
            declName = (DeclName) varDecl.getDeclName().get(0);
            circusType = (CircusType) declName.getAnn(CircusType.class);
            
            code = code + "\n" + circusType.getJavaCircusTypeName() + " " + 
                    declName + " = new " + circusType.getJavaCircusTypeName() +
                    "(RandomGenerator.generateNumber(" + 
                    circusType.getJavaCircusTypeName() + ".MIN_VALUE, " +
                    circusType.getJavaCircusTypeName() + ".MAX_VALUE" + "));";
            
            if (i == 0)
                parameters = parameters + declName;
            else
                parameters = parameters + ", " + declName;
            
        }
        
        return code;
    }

    /**
     * Transforms a signature into declarations of private attributes.
     *
     */
    public String visitingSignature(Signature signature) {
        
        String code = "";
        
        ListTerm nameTypePairs = signature.getNameTypePair();
        NameTypePair nameTypePair;
        DeclName declName;
        CircusType circusType;
        
        for (int i = 0; i < nameTypePairs.size(); i++) {
            
            nameTypePair = (NameTypePair) nameTypePairs.get(i);
            declName = nameTypePair.getName();
            circusType = (CircusType) nameTypePair.getType();
            
            code = code + "\nprivate " + circusType.getJavaCircusTypeName() + 
                    " " + declName.toString() + ";";
        }
        
        return code;
    }
}    
    
    